#!/usr/bin/env python3
"""
Rename a Hugging Face model repo by adding a suffix to its name.

Example
-------
$ python rename_hf_model.py --model-id myname/bert-base-uncased \
                            --suffix old \
                            --token $HF_TOKEN
# → renames repo to  myname/bert-base-uncased-old
"""

import argparse
import sys
from huggingface_hub import HfApi


def build_arg_parser() -> argparse.ArgumentParser:
    p = argparse.ArgumentParser(
        description="Prepend a suffix to a Hugging Face model repo name."
    )
    p.add_argument(
        "--model-id",
        required=True,
        help="Full repo ID, e.g. `your-username/bert-base-uncased`.",
    )
    p.add_argument(
        "--suffix",
        required=True,
        help="String to prepend (keep the trailing hyphen yourself if you want one).",
    )
    p.add_argument(
        "--token",
        help="🤗 Hub access token with admin rights on the repo "
        "(defaults to the HF_TOKEN env var).",
    )
    p.add_argument(
        "-y",
        "--yes",
        action="store_true",
        help="Skip the confirmation prompt.",
    )
    return p


def main() -> None:
    args = build_arg_parser().parse_args()

    # Split owner and repo name
    try:
        owner, repo_name = args.model_id.split("/", 1)
    except ValueError:
        sys.exit("❌  --model-id must be of the form <owner>/<repo_name>")

    # Compose new repo name
    new_repo_name = f"{repo_name}{args.suffix}"
    new_repo_id = f"{owner}/{new_repo_name}"

    if new_repo_id == args.model_id:
        sys.exit("⚠️  The new repo ID is identical to the current one; nothing to do.")

    # Confirm with the user
    if not args.yes:
        reply = input(
            f"About to rename '{args.model_id}' → '{new_repo_id}'. Proceed? [y/N] "
        ).lower()
        if reply != "y":
            sys.exit("Aborted by user.")

    api = HfApi()
    try:
        api.move_repo(
            from_id=args.model_id,
            to_id=new_repo_id,
            token=args.token,
            repo_type="model",
        )
    except Exception as err:
        sys.exit(f"❌  Rename failed: {err}")

    print(f"✅  Successfully renamed to {new_repo_id}")


if __name__ == "__main__":
    main()
