#!/usr/bin/env bash
set -euo pipefail

# upload_to_models.sh
# Usage: ./upload_to_models.sh models_list.txt path_to_file_or_folder

if [ "$#" -ne 2 ]; then
  echo "Usage: $0 <models_list.txt> <path_to_file_or_folder>"
  exit 1
fi

models_file="$1"
path="$2"

# Check inputs
if [ ! -f "$models_file" ]; then
  echo "Error: models list not found: $models_file"
  exit 2
fi

if [ ! -e "$path" ]; then
  echo "Error: file or folder not found: $path"
  exit 3
fi

# Make sure you're logged in
if ! huggingface-cli whoami &>/dev/null; then
  echo "You must be logged in. Run 'huggingface-cli login' first."
  exit 4
fi

# Read and upload
while IFS= read -r model; do
  # Skip empty lines or comments
  [[ -z "$model" || "$model" =~ ^# ]] && continue

  echo "→ Uploading '$path' to '$model'..."
  if huggingface-cli upload $model $path; then
    echo "✔ Success: $model"
  else
    echo "✖ Failed:  $model" >&2
  fi
done < "$models_file"
