#!/usr/bin/env bash
#
# annotate_tmux.sh  —  launch “python annotate.py <dataset>” for every dataset
# listed in a text file, each in its own tmux window.
#
# Usage:
#   ./annotate_tmux.sh datasets.txt             # reads datasets.txt, uses default session “annotate”
#   ./annotate_tmux.sh datasets.txt mysession   # custom session name
#
# The dataset list should contain one dataset per line.
# Blank lines and lines starting with ‘#’ are ignored.

set -euo pipefail

LIST_FILE="${1:-datasets.txt}"         # path to list of datasets
SESSION="${2:-annotate}"               # tmux session name (default: “annotate”)

if [[ ! -f "$LIST_FILE" ]]; then
  echo "❌  Dataset list file '$LIST_FILE' not found." >&2
  exit 1
fi

# Create (or reuse) a detached tmux session.
if ! tmux has-session -t "$SESSION" 2>/dev/null; then
  tmux new-session -d -s "$SESSION" -n "__init__"
  # tmux kill-window -t "$SESSION:__init__"        # remove placeholder window
fi

# Launch each dataset in its own window
while IFS= read -r dataset || [[ -n "$dataset" ]]; do
  # skip comments / blank lines
  [[ -z "$dataset" || "$dataset" =~ ^[[:space:]]*# ]] && continue

  # Safe window name: non-alnum → _   ;  limit length
  win="$(echo "$dataset" | tr -c '[:alnum:]' '_' | cut -c1-50)"

  tmux new-window -t "$SESSION" -n "$win" \
       "echo 'Running: python async_annotate.py \"${dataset}\"'; \
        source .venv/bin/activate; \
        python async_annotate.py \"${dataset}\"; \
        echo ''; echo 'Finished $dataset  – press q to quit this window'; \
        read -n 1 -r -p ''"
done < "$LIST_FILE"

echo "✅  All jobs started in tmux session '$SESSION'."
echo "   Attach with:  tmux attach -t $SESSION"
