import argparse
import mteb
from mteb.benchmarks import Benchmark
from mteb.overview import get_tasks
import logging

# Set logging level for MTEB
logging.getLogger("mteb").setLevel(logging.INFO)

# === Custom Benchmark ===
smolmieb = Benchmark(
    name="MIEB(smol)",
    tasks=get_tasks(
        tasks=[
        # Document retrieval.
        "VidoreArxivQARetrieval",
        "VidoreDocVQARetrieval",
        "VidoreInfoVQARetrieval",
        "VidoreTabfquadRetrieval",
        "VidoreTatdqaRetrieval",
        "VidoreShiftProjectRetrieval",
        "VidoreSyntheticDocQAAIRetrieval",
        "VidoreSyntheticDocQAEnergyRetrieval",
        "VidoreSyntheticDocQAGovernmentReportsRetrieval",
        "VidoreSyntheticDocQAHealthcareIndustryRetrieval",
        "Vidore2ESGReportsRetrieval",
        "Vidore2EconomicsReportsRetrieval",
        "Vidore2BioMedicalLecturesRetrieval",
        "Vidore2ESGReportsHLRetrieval",
        # Caption retrieval.
        "MSCOCOT2IRetrieval",
        "MSCOCOI2TRetrieval",
        "Flickr30kT2IRetrieval",
        "Flickr30kI2TRetrieval",
        # Image class.
        "Caltech101",
        "StanfordCars",
        "OxfordPets",
        "Food101Classification",
        "Imagenet1k",
        ],
    ),
    description="""MIEB(lite) is a comprehensive image embeddings benchmark, spanning 10 task types, covering 51 tasks.
    This is a lite version of MIEB(Multilingual), designed to be run at a fraction of the cost while maintaining
    relative rank of models.""",
    reference="https://arxiv.org/abs/2504.10471",
    contacts=["gowitheflow-1998", "isaac-chung"],
    citation=r"""
@article{xiao2025mieb,
  author = {Chenghao Xiao and Isaac Chung and Imene Kerboua and Jamie Stirling and Xin Zhang and Márton Kardos and Roman Solomatin and Noura Al Moubayed and Kenneth Enevoldsen and Niklas Muennighoff},
  doi = {10.48550/ARXIV.2504.10471},
  journal = {arXiv preprint arXiv:2504.10471},
  publisher = {arXiv},
  title = {MIEB: Massive Image Embedding Benchmark},
  url = {https://arxiv.org/abs/2504.10471},
  year = {2025},
}
""",
)

def parse_args():
    parser = argparse.ArgumentParser(description="Run MTEB benchmarks on a given model")
    parser.add_argument(
        "--model-name", "-m",
        type=str,
        required=True,
        help="Name of the model to load via mteb.get_model"
    )
    parser.add_argument(
        "--batch-size", "-b",
        type=int,
        default=32,
        help="Batch size for encoding tasks"
    )
    parser.add_argument(
        "--output-folder", "-o",
        type=str,
        default="results",
        help="Folder to save the results"
    )
    return parser.parse_args()

def main():
    args = parse_args()

    model = mteb.get_model(
        model_name=args.model_name,
    )
    evaluator = mteb.MTEB(tasks=smolmieb)
    results_test = evaluator.run(
        model,
        output_folder=args.output_folder,
        verbosity=3,
        encode_kwargs={"batch_size": args.batch_size}
    )

if __name__ == "__main__":
    main()
