#!/usr/bin/env bash
# batch_rename_model.sh
#
# Rename every model listed in <models.txt> by prepending <suffix>.
# Extra flags (‒‒token, -y, …) are forwarded to rename_hf_model.py.
#
# Example
#   ./batch_rename_model.sh models.txt finetuned --token "$HF_TOKEN" -y

set -euo pipefail

usage() {
  echo "Usage: $0 <model-list.txt> <suffix> [additional options for rename_hf_model.py]" >&2
  exit 1
}

[[ $# -lt 2 ]] && usage

LISTFILE=$1        # Path to the text file with one model id per line
SUFFIX=$2          # String to prepend
shift 2            # Remaining CLI args (token, -y, …) get passed through

REPO_PATH=$(dirname "$(realpath "$0")")
PY_SCRIPT="${REPO_PATH}/rename_model.py"  # Adjust if the script lives elsewhere

[[ ! -f $LISTFILE ]] && { echo "❌  File '$LISTFILE' not found." >&2; exit 1; }

while IFS= read -r MODEL_ID || [[ -n $MODEL_ID ]]; do
  # Skip blank lines and comments that start with '#'
  [[ -z $MODEL_ID || ${MODEL_ID:0:1} == "#" ]] && continue

  echo "▶ Renaming $MODEL_ID …"
  python "$PY_SCRIPT" --model-id "$MODEL_ID" --suffix "$SUFFIX" -y "$@"
done < "$LISTFILE"
