#!/usr/bin/env python3
"""
Print the latest commit hash for each Hugging Face model
listed in a .txt file (one model ID per line).

Usage:
    python latest_hf_commit.py models.txt

Requires:
    pip install huggingface_hub
    (Optionally set HF_HUB_TOKEN in your environment for private models or higher rate limits)
"""

import sys
from huggingface_hub import HfApi

def main(models_file: str):
    api = HfApi()  # will pick up HF_HUB_TOKEN or local CLI login if available

    try:
        with open(models_file, 'r') as f:
            models = [line.strip() for line in f if line.strip() and not line.startswith('#')]
    except FileNotFoundError:
        print(f"[ERROR] File not found: {models_file}", file=sys.stderr)
        sys.exit(1)

    for model_id in models:
        try:
            # list_repo_commits returns commits sorted by date (newest first) :contentReference[oaicite:0]{index=0}
            commits = api.list_repo_commits(repo_id=model_id, repo_type="model")
            if commits:
                latest = commits[0].commit_id  # GitCommitInfo.commit_id is the SHA :contentReference[oaicite:1]{index=1}
                print(f"- {model_id}: {latest}")
            else:
                print(f"- {model_id}: [no commits found]")
        except Exception as e:
            print(f"- {model_id}: [ERROR: unexpected error: {e}]")

if __name__ == "__main__":
    if len(sys.argv) != 2:
        print("Usage: python latest_hf_commit.py <models.txt>", file=sys.stderr)
        sys.exit(1)
    main(sys.argv[1])
