import torch
from torch import nn
from arithmetic_mod7.data import token_ids


class ResBlock(nn.Module):
    def __init__(self):
        super().__init__()
        self.conv1 = nn.Conv1d(32, 32, 3, 1, 1)
        self.conv2 = nn.Conv1d(32, 32, 3, 1, 1)
        self.relu = nn.ReLU()

    def forward(self, x, mask=None):
        if mask is not None:
            x = x * mask
        h = self.conv1(x)
        h = self.relu(h)
        if mask is not None:
            h = h * mask
        h = self.conv2(h)
        h = self.relu(h)
        return x + h


class Encoder(nn.Module):
    def __init__(self, from_dim, to_dim, dim_per_head, n_head):
        super().__init__()
        out_dim = dim_per_head * n_head
        self.out_dim = out_dim
        self.dim_per_head = dim_per_head
        self.n_head = n_head

        self.rpos_fc = nn.Linear(1, out_dim)
        self.rpos_fc2 = nn.Linear(out_dim, out_dim)
        self.relu = nn.ReLU()

        self.from_fc = nn.Linear(from_dim, out_dim)
        self.to_fc = nn.Linear(to_dim, 2 * out_dim)
        self.ctx_fc = nn.Linear(out_dim, from_dim)

        self.ffd_fc1 = nn.Linear(from_dim, 4 * from_dim)
        self.ffd_fc2 = nn.Linear(4 * from_dim, from_dim)
        self.activation = nn.GELU()
        self.ln1 = nn.LayerNorm(from_dim)
        self.ln2 = nn.LayerNorm(from_dim)

    def forward(self, x, y, mask):
        B, F, _ = x.shape
        _, T, _ = y.shape
        N, D = self.n_head, self.dim_per_head

        q = self.from_fc(x)    # B, F, C
        kv = self.to_fc(y)    # B, T, C
        k, v = torch.split(kv, [N * D, N * D], dim=-1)   # B, T, C
        q = q.view((B, F, N, D))
        k = k.view((B, T, N, D))
        v = v.view((B, T, N, D))
        score = torch.einsum("bfnd,btnd->bnft", q, k)   # B, N, F, T

        orig_rpos = 0.01 * (torch.arange(0, F)[:, None] - torch.arange(0, T)[None, :])  # F, T
        rpos = self.rpos_fc2(self.relu(self.rpos_fc(orig_rpos[:, :, None])))   # F, T, C
        rpos = rpos.reshape((F, T, N, D))  # F, T, N, D
        score_rpos = torch.einsum("bfnd,ftnd->bnft", q, rpos)
        score = score + score_rpos

        score = score / D ** 0.5
        if mask is not None:
            score = score - (1.0 - mask.float()[:, None, None, :]) * 1e8
        four_R_plus_1_mask = (orig_rpos < 8.5 / 100).float() * (orig_rpos > -8.5 / 100).float()   # avoid float error
        # print(four_R_plus_1_mask)
        score = score - (1.0 - four_R_plus_1_mask[None, None, :, :]) * 1e8

        prob = torch.softmax(score, dim=-1)   # B, N, F, T
        context = torch.einsum("bnft,btnd->bfnd", prob, v)   # B, F, N, D

        context = context.reshape((B, F, N * D))
        x = self.ln1(x + self.ctx_fc(context))

        ffd = self.ffd_fc2(self.activation(self.ffd_fc1(x)))
        x = self.ln2(x + ffd)

        return x, prob


class ParseModel(nn.Module):
    def __init__(self, vocab_size):
        super().__init__()

        self._vocab_size = vocab_size

        self.embed = nn.Linear(vocab_size, 32, bias=False)
        self.enc1 = Encoder(32, 32, 32, 4)
        self.enc2 = Encoder(32, 32, 32, 4)
        self.enc3 = Encoder(32, 32, 32, 4)
        # self.res1 = ResBlock()
        # self.res2 = ResBlock()
        # self.res3 = ResBlock()
        # self.res4 = ResBlock()
        # self.res5 = ResBlock()
        # self.res6 = ResBlock()
        self.out = nn.Linear(32, len(token_ids))

    def forward(self, x, mask=None, y=None):
        h = self.embed(x)   # B, T, C
        oh = h
        h, _ = self.enc1(h, oh, mask)
        h, _ = self.enc2(h, oh, mask)
        h, _ = self.enc3(h, oh, mask)
        # h = torch.transpose(h, 1, 2)
        # h = self.res1(h, mask[:, None, :])
        # h = self.res2(h, mask[:, None, :])
        # h = self.res3(h, mask[:, None, :])
        # h = self.res4(h, mask[:, None, :])
        # h = self.res5(h, mask[:, None, :])
        # h = self.res6(h, mask[:, None, :])
        # h = torch.transpose(h, 1, 2)
        out = self.out(h)   # B, T, n
        dist = torch.distributions.Categorical(logits=out)

        if y is not None:
            loss = -dist.log_prob(y) * mask
            return out, dist, loss
        return out, dist
