import os
import glob
import imageio
from PIL import Image, ImageDraw, ImageFont
from natsort import natsorted


def find_files(folder_path):
    ## Use glob to search for PNG files in the specified folder
    png_files = glob.glob(os.path.join(folder_path, '*.png'))
    ## Sort the list of PNG files numerically, so eg prefix_10_suffix goes after prefix_5_suffix
    png_files = natsorted(png_files)
    for file in png_files:
        print(file)
    return png_files

def animate(folder_path, frames, out_file_name='anim', duration=.05, loop=0):
    """
    Finds all pngs in the specified folder_path,
    orders them numerically and stores anim.gif to the same folder.
    """
    print(f'Read {len(frames)} frames')
    # frame_durations = [duration for f in frames]
    # imageio.mimsave(f'{folder_path}/{out_file_name}.gif', frames, duration=duration, loop=loop)
    imageio.mimwrite(f'{folder_path}/{out_file_name}.gif', frames, 'GIF', duration=duration, loop=loop)
    # imageio.mimsave(f'{folder_path}/anim.gif', frames, duration=duration, loop=loop)
    print(f'GIF saved!')

def add_text_to_img(img, text, position, font_size, color='black'):
    draw = ImageDraw.Draw(img)
    font = ImageFont.truetype("arial.ttf", font_size)
    draw.text(position, text, font=font, fill=color)  # You can change the color as needed
    return img


def create_progress_bar(progress):
    bar_length = 46
    block = int(round(bar_length * progress))
    progress_bar = f"{'█' * block}{'░' * (bar_length - block)} {progress * 100:.0f}%"
    return progress_bar

if __name__ == '__main__':
    # folder_path = "img/random/Boundary"
    # folder_path = 'media/images'
    # folder_path = r'img\general_net\Boundary'
    # folder_path = r'img\general_net\Trajectories of surfnet graph edges'

    # folder_path = r'C:\Users\admin\phd\projects\23-objgen\frames_over_training\frames'
    folder_path = r'C:\Users\admin\phd\projects\23-objgen\4ag1a9cf_frames\4ag1a9cf'
    dur = 0.03

    use_progress_bar = True

    if use_progress_bar:
        crop_area = (310, 100, 1640, 1100)
    else:
        crop_area = (310, 100, 1640, 1020)
    png_files = find_files(folder_path)
    # quite fast and 20MB
    # schedule = [(0, 5), (100, 10), (500, 50), (800, 100)]
    # schedule = [(0, 5), (100, 10), (500, 20)]
    # schedule = [(0, 1000)]
    ## NICE
    schedule = [(0, 2), (100, 4), (200, 8), (500, 25), (1000, 50)]
    # schedule = [(0, 5)]  # too uninteresting at the end; 401 frames: 102MB; compressed: 38.5 --> aim for 156 frames
    # slower
    # schedule = [(0, 1), (100, 5), (500, 20), (1000, 50)]

    png_files = png_files[:2000]
    stop_at = len(png_files) - 1
    it_stop_at = '2000'

    frames = []
    cur_bucket = 0
    for i, png_file in enumerate(png_files):
        if cur_bucket < len(schedule) - 1 and i >= schedule[cur_bucket + 1][0]:
            cur_bucket += 1

        if i % schedule[cur_bucket][1] != 0 and i != stop_at:
            continue

        file_name = png_file.split(os.path.sep)[-1].split('.png')[0]
        ## convert to 0-based idcs instead of 1-based
        file_name = str(int(file_name) - 1)
        if i == stop_at:
            file_name = it_stop_at

        frame = Image.open(png_file)
        # print(f'img size: {frame.size}')
        frame = frame.crop(crop_area)
        ## add white background as the transparent background is not satisfactory
        white_background = Image.new("RGBA", frame.size, "WHITE")
        white_background.paste(frame, (0, 0), frame)
        rgb_image = white_background.convert("RGB")
        ## add frame text
        frame_w_text = add_text_to_img(rgb_image, f'it: {file_name}', (20, 20), 36)

        if use_progress_bar:
            progr = int(file_name) * 1.0 / len(png_files)
            progr_str = create_progress_bar(progr)
            # frame_w_text = add_text_to_img(frame_w_text, f'{progr_str}', (frame_w_text.size[0] // 2 - 70, frame_w_text.size[1] - 70), 36)
            frame_w_text = add_text_to_img(frame_w_text, f'{progr_str}', (20, frame_w_text.size[1] - 70), 36)

        print(f'append file {file_name}')
        frames.append(frame_w_text)

    DEFAULT_DUR = 30
    dur = [DEFAULT_DUR for frame in frames]  # in milliseconds
    dur[-1] = 2000.0
    animate(folder_path, frames, f'jeb_training-pb:{use_progress_bar}-{DEFAULT_DUR}-{schedule}', dur, loop=0)