import numpy as np
from typing import Optional, Callable
from scipy.optimize import minimize


class OptResult(object):
    def __init__(self,
                 params: np.ndarray,
                 value: float,
                 success: bool,
                 status: int,
                 message: str = ""):
        """
        Result of optimization over parameters.
        :param params: np.ndarray, the optimal parameters
        :param value: float, the optimal objective value
        :param success: bool, true if optimization succeeded, else false
        :param message: str, messaging indicating status of optimization
        :param status: int, termination status
        for 'trust-constr' method, 0 : The maximum number of function evaluations is exceeded.
        1 : gtol termination condition is satisfied. 2 : xtol termination condition is satisfied. 
        3 : callback function requested termination.
        for 'BFGS' method:
        0: The optimization terminated successfully.
        1: The maximum number of function evaluations was exceeded.
        2: The algorithm did not converge to a solution within the specified tolerance.
        3: The line search algorithm could not find a better solution.
        4: The optimization was terminated by the user.
        5: The optimization encountered an error.
        """
        self.params = params
        self.value = value
        self.success = success
        self.status = status
        self.message = message
        

class ScipyMinimizer():
    def __init__(self,
                 method: str = 'BFGS',
                 tol: float = 5e-03,
                 options: Optional[dict] = None):
        """
        Specific minimizer which simply wraps the scipy.optimize.minimize method for convenience
        For more details see:
            https://docs.scipy.org/doc/scipy/reference/generated/scipy.optimize.minimize.html
        :param method: str, which method to pass to optimizer
        :param tol: float, the function tolerance
        :param options: dict, options to pass to scipy.optimize.minimize
        """
        self._method = method  # optimization method to use
        self._tol = tol
        # self._options = options or {'maxiter': 500, 'gtol': 1e-03, 'xtol': 1e-03, 'verbose': 1} # For 'trust-constr' method
        self._options = options or {'disp':True, 'maxiter': 500, 'gtol': 1e-02} # For 'BFGS' method

    def minimize(self,
                 function: Callable,
                 guess: Optional[np.ndarray] = None) -> OptResult:
        """
        Minimize the objectives to obtain optimal params. Main function to override
        :param function: Callable, the objective function to minimizer
        :param guess: np.ndarray, initial guess of parameters
        :return: OptResult, the result of optimization
        """
        res = minimize(function,
                       guess,
                       tol=self._tol,
                       method=self._method,
                       options=self._options)

        return OptResult(params=res.x, value=res.fun, success=res.success, status=res.status, message=res.message)