import os

import numpy as np
import torch
import torch.nn as nn
import torch.nn.functional as F
import torch.optim as optim

from meta_neural_network_architectures import VGGReLUNormNetwork
from inner_loop_optimizers import LSLRGradientDescentLearningRule

from copy import copy, deepcopy
from cca import CCAHook
import cca_core
import pwcca
import time
#from anatome import CCAHook

def set_torch_seed(seed):
    """
    Sets the pytorch seeds for current experiment run
    :param seed: The seed (int)
    :return: A random number generator to use
    """
    rng = np.random.RandomState(seed=seed)
    torch_seed = rng.randint(0, 999999)
    torch.manual_seed(seed=torch_seed)

    return rng


class MAMLFewShotClassifier(nn.Module):
    def __init__(self, im_shape, device, args):
        """
        Initializes a MAML few shot learning system
        :param im_shape: The images input size, in batch, c, h, w shape
        :param device: The device to use to use the model on.
        :param args: A namedtuple of arguments specifying various hyperparameters.
        """
        super(MAMLFewShotClassifier, self).__init__()
        self.args = args
        self.device = device
        self.batch_size = args.batch_size
        self.use_cuda = args.use_cuda
        self.im_shape = im_shape
        self.current_epoch = 0
        
        self.num_tasks=self.args.num_train_tasks
        self.lr_p = self.args.p_lr #0.00001 #0.00002 for 5 way
        self.p= np.ones(self.num_tasks)/self.num_tasks
        if not self.args.TR_MAML:
            self.TR_MAML = False
        else:
            self.TR_MAML = True
        self.count = 0
        self.max = 0
        self.old_max = 9
        self.old_val_max = 3
        self.ERM = self.args.ERM

        self.rng = set_torch_seed(seed=args.seed)
        self.classifier = VGGReLUNormNetwork(im_shape=self.im_shape, num_output_classes=100, #self.args.num_classes_per_set,
                                             args=args, device=device, meta_classifier=True).to(device=self.device)
        self.task_learning_rate = args.task_learning_rate

        self.old_epoch = -1
        self.inner_loop_optimizer = LSLRGradientDescentLearningRule(device=device,
                                                                    init_learning_rate=1*self.task_learning_rate,
                                                                    total_num_inner_loop_steps=self.args.number_of_training_steps_per_iter,
                                                                    use_learnable_learning_rates=self.args.learnable_per_layer_per_step_inner_loop_learning_rate)
        self.inner_loop_optimizer.initialise(
            names_weights_dict=self.get_inner_loop_parameter_dict(params=self.classifier.named_parameters()))

        print("Inner Loop parameters")
        for key, value in self.inner_loop_optimizer.named_parameters():
            print(key, value.shape)

        self.use_cuda = args.use_cuda
        self.device = device
        self.args = args
        self.to(device)
        print("Outer Loop parameters")
        for name, param in self.named_parameters():
            if param.requires_grad:
                print(name, param.shape, param.device, param.requires_grad)


        self.optimizer = optim.Adam(self.trainable_parameters(), lr=args.meta_learning_rate, amsgrad=False)
        self.scheduler = optim.lr_scheduler.CosineAnnealingLR(optimizer=self.optimizer, T_max=self.args.total_epochs,
                                                              eta_min=self.args.min_learning_rate)

        self.device = torch.device('cpu')
        if torch.cuda.is_available():
            if torch.cuda.device_count() > 1:
                self.to(torch.cuda.current_device())
                self.classifier = nn.DataParallel(module=self.classifier)
            else:
                self.to(torch.cuda.current_device())

            self.device = torch.cuda.current_device()

    def get_classifier(self):
        return self.classifier

    def get_per_step_loss_importance_vector(self):
        """
        Generates a tensor of dimensionality (num_inner_loop_steps) indicating the importance of each step's target
        loss towards the optimization loss.
        :return: A tensor to be used to compute the weighted average of the loss, useful for
        the MSL (Multi Step Loss) mechanism.
        """
        loss_weights = np.ones(shape=(self.args.number_of_training_steps_per_iter)) * (
                1.0 / self.args.number_of_training_steps_per_iter)
        decay_rate = 1.0 / self.args.number_of_training_steps_per_iter / self.args.multi_step_loss_num_epochs
        min_value_for_non_final_losses = 0.03 / self.args.number_of_training_steps_per_iter
        for i in range(len(loss_weights) - 1):
            curr_value = np.maximum(loss_weights[i] - (self.current_epoch * decay_rate), min_value_for_non_final_losses)
            loss_weights[i] = curr_value

        curr_value = np.minimum(
            loss_weights[-1] + (self.current_epoch * (self.args.number_of_training_steps_per_iter - 1) * decay_rate),
            1.0 - ((self.args.number_of_training_steps_per_iter - 1) * min_value_for_non_final_losses))
        loss_weights[-1] = curr_value
        loss_weights = torch.Tensor(loss_weights).to(device=self.device)
        return loss_weights

    def get_inner_loop_parameter_dict(self, params):
        """
        Returns a dictionary with the parameters to use for inner loop updates.
        :param params: A dictionary of the network's parameters.
        :return: A dictionary of the parameters to use for the inner loop optimization process.
        """
        param_dict = dict()
        for name, param in params:
            if param.requires_grad:
                if self.args.enable_inner_loop_optimizable_bn_params:
                    param_dict[name] = param.to(device=self.device)
                else:
                    if "norm_layer" not in name:
                        param_dict[name] = param.to(device=self.device)

        return param_dict

    def apply_inner_loop_update(self, loss, names_weights_copy, use_second_order, current_step_idx):
        """
        Applies an inner loop update given current step's loss, the weights to update, a flag indicating whether to use
        second order derivatives and the current step's index.
        :param loss: Current step's loss with respect to the support set.
        :param names_weights_copy: A dictionary with names to parameters to update.
        :param use_second_order: A boolean flag of whether to use second order derivatives.
        :param current_step_idx: Current step's index.
        :return: A dictionary with the updated weights (name, param)
        """
        num_gpus = torch.cuda.device_count()
        if num_gpus > 1:
            self.classifier.module.zero_grad(params=names_weights_copy)
        else:
            self.classifier.zero_grad(params=names_weights_copy)

        grads = torch.autograd.grad(loss, names_weights_copy.values(),
                                    create_graph=use_second_order, allow_unused=True)
        names_grads_copy = dict(zip(names_weights_copy.keys(), grads))

        names_weights_copy = {key: value[0] for key, value in names_weights_copy.items()}

        for key, grad in names_grads_copy.items():
            if grad is None:
                print('Grads not found for inner loop parameter', key)
            names_grads_copy[key] = names_grads_copy[key].sum(dim=0)

        names_weights_copy = self.inner_loop_optimizer.update_params(names_weights_dict=names_weights_copy,
                                                                     names_grads_wrt_params_dict=names_grads_copy,
                                                                     num_step=1)#current_step_idx)

        num_devices = torch.cuda.device_count() if torch.cuda.is_available() else 1
        names_weights_copy = {
            name.replace('module.', ''): value.unsqueeze(0).repeat(
                [num_devices] + [1 for i in range(len(value.shape))]) for
            name, value in names_weights_copy.items()}

        return names_weights_copy

    def get_across_task_loss_metrics(self, total_losses, total_accuracies):
        losses = dict()

    #    print("TOTAL_ACCS")
     #   print(total_accuracies)

        losses['loss'] = torch.mean(torch.stack(total_losses))
        losses['mx_loss'] = torch.max(torch.stack(total_losses))
        losses['std_loss'] = torch.std(torch.stack(total_losses))
        losses['accuracy'] = torch.mean(torch.stack(total_accuracies))
#        losses['min_accuracy'] = np.min(total_accuracies)
#        losses['std_accuracy'] = np.std(total_accuracies)
#         losses['full_loosses'] = torch.stack(total_losses) # need to makes ure right datatype

        return losses

    def forward(self, data_batch, epoch, use_second_order, use_multi_step_loss_optimization, num_steps, training_phase,hook):
        """
        Runs a forward outer loop pass on the batch of tasks using the MAML/++ framework.
        :param data_batch: A data batch containing the support and target sets.
        :param epoch: Current epoch's index
        :param use_second_order: A boolean saying whether to use second order derivatives.
        :param use_multi_step_loss_optimization: Whether to optimize on the outer loop using just the last step's
        target loss (True) or whether to use multi step loss which improves the stability of the system (False)
        :param num_steps: Number of inner loop steps.
        :param training_phase: Whether this is a training phase (True) or an evaluation phase (False)
        :return: A dictionary with the collected losses of the current outer forward propagation.
        """
        x_support_set, x_target_set, y_support_set, y_target_set, task_ind_set = data_batch

        [b, ncs, spc] = y_support_set.shape

        self.num_classes_per_set = ncs

        total_losses = []
        total_accuracies = []
        per_task_target_preds = [[] for i in range(len(x_target_set))]
        self.classifier.zero_grad()
        dists = []
        # run a ton of data thru current network
        # save activations
        if hook:
        #    self.classifier.get_activations(data)

            names_weights_copy = self.get_inner_loop_parameter_dict(self.classifier.named_parameters())

            num_devices = torch.cuda.device_count() if torch.cuda.is_available() else 1

            names_weights_copy = {
                name.replace('module.', ''): value.unsqueeze(0).repeat(
                    [num_devices] + [1 for i in range(len(value.shape))]) for
                name, value in names_weights_copy.items()}

            layers = self.classifier.get_layer_keys()
            hooks = [CCAHook(self.classifier, 'layer_dict.'+name, svd_device=self.device) for name in layers]

           # print("DATA SIZE")
           # print(len(x_target_set))
            #print(x_target_set)
            
            for (i,xtask) in enumerate(x_target_set):
                n, s, c, h, w = xtask.shape
                xtask = xtask.view(-1, c, h, w)
                if i == 0:
                    inp = xtask
                else:
                    inp = torch.cat((inp,xtask))

            for xtask in x_support_set:
                n, s, c, h, w = xtask.shape
                xtask = xtask.view(-1, c, h, w)
                inp = torch.cat((inp,xtask))

            preds = self.classifier.forward(x=inp, params=names_weights_copy,
                                        training=True,
                                        backup_running_statistics=True,num_step=0)
            activs1 = [h.get_hooked_value() for h in hooks]
            #print("ACTIVS1LEN")
           # print(len(activs1))
          #  print(activs1[1].shape)
         #   print(activs1[2].shape)
            self.classifier.restore_backup_stats()
        
        hard = []
        easy = []
        hard_acc = []
        easy_acc = []
        for task_id, (x_support_set_task, y_support_set_task, x_target_set_task, y_target_set_task,task_ind) in \
                enumerate(zip(x_support_set,
                              y_support_set,
                              x_target_set,
                              y_target_set,task_ind_set)):
            task_losses = []
            task_accuracies = []
            per_step_loss_importance_vectors = self.get_per_step_loss_importance_vector()
            names_weights_copy = self.get_inner_loop_parameter_dict(self.classifier.named_parameters())

            num_devices = torch.cuda.device_count() if torch.cuda.is_available() else 1

            names_weights_copy = {
                name.replace('module.', ''): value.unsqueeze(0).repeat(
                    [num_devices] + [1 for i in range(len(value.shape))]) for
                name, value in names_weights_copy.items()}

          #  nwc_old = copy(names_weights_copy)
            n, s, c, h, w = x_target_set_task.shape

            x_support_set_task = x_support_set_task.view(-1, c, h, w)
            y_support_set_task = y_support_set_task.view(-1)
            x_target_set_task = x_target_set_task.view(-1, c, h, w)
            y_target_set_task = y_target_set_task.view(-1)

         #   print(names_weights_copy)
        #    if hook:
         #       print(names_weights_copy)
                # TO DO: use a lot of data. somehow measure similarity of layers. cook.
                # save old network somehow
                # update current network with gd
                # 

                #classi1 = self.classifier
          #      classi2 = deepcopy(self.classifier)#self.model.get_classifier()
           #     layers = self.classifier.get_layer_keys()
             #   hooks1 = [CCAHook(classi1, name, svd_device=self.device) for name in layers]
               # hooks2 = [CCAHook(classi2, name, svd_device=self.device) for name in layers]
                
 
 #           preds = classi2.forward(x=x_support_set_task, params=names_weights_copy,
  #                                      training=True,
   #                                     backup_running_statistics=True if (num_step == 0) else False,num_step=num_step)
            fixed = False
            if not training_phase:
                num_steps = self.args.number_of_evaluation_steps_per_iter
            for num_step in range(num_steps):

                support_loss, support_preds = self.net_forward(x=x_support_set_task,
                                                               y=y_support_set_task,
                                                               weights=names_weights_copy,
                                                               backup_running_statistics=
                                                               True if (num_step == 0) else False,
                                                               training=True, num_step=num_step)
                if self.ERM:
                    inner_steps = 0
                else:
                    inner_steps = 1
                if (num_step<inner_steps) or not training_phase:
                    names_weights_copy = self.apply_inner_loop_update(loss=support_loss,
                                                                  names_weights_copy=names_weights_copy,
                                                                  use_second_order=use_second_order,
                                                                  current_step_idx=num_step)

                if use_multi_step_loss_optimization and training_phase and epoch < self.args.multi_step_loss_num_epochs:
                    target_loss, target_preds = self.net_forward(x=x_target_set_task,
                                                                 y=y_target_set_task, weights=names_weights_copy,
                                                                 backup_running_statistics=False, training=True,
                                                                 num_step=num_step)

                    task_losses.append(per_step_loss_importance_vectors[num_step] * target_loss)
                else:
                    if num_step == (self.args.number_of_training_steps_per_iter - 1):
                        target_loss, target_preds = self.net_forward(x=x_target_set_task,
                                                                     y=y_target_set_task, weights=names_weights_copy,
                                                                     backup_running_statistics=False, training=True,
                                                                     num_step=num_step)
                        task_losses.append(target_loss)

            per_task_target_preds[task_id] = target_preds.detach().cpu().numpy()
            _, predicted = torch.max(target_preds.data, 1)
            #if hook:
       #         print('NAMED_MODULES')
        #        print(self.classifier.named_modules)
         #       _dict = {n: m for n, m in self.classifier.named_modules()}
          #      print(_dict.keys())
 
             #   hooks1 = [CCAHook(self.classifier, 'layer_dict.'+name) for name in layers]
              #  hooks2 = [CCAHook(classi2, 'layer_dict.'+name) for name in layers]

             #   preds = self.classifier.forward(x=x_support_set_task, params=names_weights_copy,
              #                          training=True,
               #                         backup_running_statistics=True if (num_step == 0) else False,num_step=num_step)
            
               # preds = classi2.forward(x=x_support_set_task, params=nwc_old,
                #                        training=True,
                 #                       backup_running_statistics=True if (num_step == 0) else False,num_step=num_step)
               # history = [h1.distance(h2) for h1, h2 in zip(hooks1, hooks2)]               
                #dists.append(history)
    #        print("PREDS")
     #       print(target_preds.data)
      #      print("PREDICTIONS")
       #     print(predicted)
        #    print("TRUE LABELS")
         #   print(y_target_set_task.data.float())

            accuracy = predicted.float().eq(y_target_set_task.data.float()).cpu().float()
            task_losses = torch.sum(torch.stack(task_losses))
            total_losses.append(task_losses)
            total_accuracies.extend(accuracy)
#            print((predicted,y_target_set_task.data, task_ind))
 #           time.sleep(1)
            
            if hook and 0:
                preds = self.classifier.forward(x=inp, params=names_weights_copy,
                                            training=True,
                                            backup_running_statistics=True if (num_step == 0) else False,num_step=num_step)
                activs2 = [h.get_hooked_value() for h in hooks]

                #  print(len(activs2))
                # print(activs2[0].shape)
                #postprocess
                #dists = cca_core.get_cca_similarity(activs1, activs2, verbose=True)

                # pwcca_mean, w, _ = pwcca.compute_pwcca(activs1, activs2, epsilon=1e-10)
                for (acts1, acts2) in zip(activs1[4], activs2[4]):
                  #  print(acts1.shape)
                    #n, c = acts1.shape
                    f_acts1 = activs1[4]#.reshape((n*h*w, c))
                    f_acts2 = activs2[4]#.reshape((n*h*w, c))
                    pwcca_mean, w, _ = pwcca.compute_pwcca(f_acts1.T, f_acts2.T,epsilon=1e-10)#[:,::2], f_acts2.T[:,::2], epsilon=1e-10)
                    if task_ind == 0:
                        easy.append(pwcca_mean)
                    else:
                        hard.append(pwcca_mean)
                   # dists.append(pwcca_mean)

                if task_ind == 0:
                    easy_acc.append(torch.mean(accuracy))
                else:
                    hard_acc.append(torch.mean(accuracy))

            #accuracy = predicted.float().eq(y_target_set_task.data.float()).cpu().float()
            #task_losses = torch.sum(torch.stack(task_losses))
            #total_losses.append(task_losses)
            #total_accuracies.extend(accuracy)

            if not training_phase:
                self.classifier.restore_backup_stats()

        # pass a bunch of data through the new network
        # save the activations
        # call cca function
        # get one number , compare it to the number obtained when taking g steps on easy tasks
        if hook:
           # preds = self.classifier.forward(x=inp, params=names_weights_copy,
            #                            training=True,
             #                           backup_running_statistics=True if (num_step == 0) else False,num_step=num_step)
           # activs2 = [h.get_hooked_value() for h in hooks]
            
          #  print(len(activs2))
           # print(activs2[0].shape)
            #postprocess
            #dists = cca_core.get_cca_similarity(activs1, activs2, verbose=True)
            
           # pwcca_mean, w, _ = pwcca.compute_pwcca(activs1, activs2, epsilon=1e-10)
           # for (acts1, acts2) in zip(activs1[:4], activs2[:4]):
               # print(acts1.shape)
               # n, c, h, w = acts1.shape
              #  f_acts1 = acts1.reshape((n*h*w, c))
             #   f_acts2 = acts2.reshape((n*h*w, c))
            #    pwcca_mean, w, _ = pwcca.compute_pwcca(f_acts1.T[:,::2], f_acts2.T[:,::2], epsilon=1e-10)
           #     dists.append(pwcca_mean)
            easy_avg = sum(easy)/len(easy)
            hard_avg = sum(hard)/len(hard)
            easy_acc_avg = sum(easy_acc)/len(easy_acc)
            hard_acc_avg = sum(hard_acc)/len(hard_acc)
            print("easy:")
            print(easy_avg)
            print(easy_acc_avg)
            print("hard:")
            print(hard_avg)
            print(hard_acc_avg)

            # todo: data, .get_value(), postprocess, when to compute activs2
        losses = self.get_across_task_loss_metrics(total_losses=total_losses,
                                                   total_accuracies=total_accuracies)

        for idx, item in enumerate(per_step_loss_importance_vectors):
            losses['loss_importance_vector_{}'.format(idx)] = item.detach().cpu().numpy()

       # print(losses)
       # print(total_accuracies)
        return losses, total_losses, total_accuracies, per_task_target_preds, dists

  #  def net_forward(self, x, y, weights, backup_running_statistics, training, num_step):
        """
        A base model forward pass on data points x. Using the parameters in the weights dictionary. Also requires
        boolean flags indicating whether to reset the running statistics at the end of the run (if at evaluation phase).
        A flag indicating whether this is the training session and an int indicating the current step's number in the
        inner loop.
        :param x: A data batch of shape b, c, h, w
        :param y: A data targets batch of shape b, n_classes
        :param weights: A dictionary containing the weights to pass to the network.
        :param backup_running_statistics: A flag indicating whether to reset the batch norm running statistics to their
         previous values after the run (only for evaluation)
        :param training: A flag indicating whether the current process phase is a training or evaluation.
        :param num_step: An integer indicating the number of the step in the inner loop.
        :return: the crossentropy losses with respect to the given y, the predictions of the base model.
        """
        #preds = self.classifier.forward(x=x, params=weights,
       #                                 training=training,
      #                                  backup_running_statistics=backup_running_statistics, num_step=num_step)

     #   loss = F.cross_entropy(input=preds, target=y)

    #    return loss, preds

   # def trainable_parameters(self):
        """
        Returns an iterator over the trainable parameters of the model.
        """
      #  for param in self.parameters():
     #       if param.requires_grad:
    #            yield param

   # def train_forward_prop(self, data_batch, epoch):
        """
        Runs an outer loop forward prop using the meta-model and base-model.
        :param data_batch: A data batch containing the support set and the target set input, output pairs.
        :param epoch: The index of the currrent epoch.
        """
  #      for idx, item in enumerate(per_step_loss_importance_vectors):
 #           losses['loss_importance_vector_{}'.format(idx)] = item.detach().cpu().numpy()

#        return losses, total_losses, total_accuracies, per_task_target_preds, dists

    def net_forward(self, x, y, weights, backup_running_statistics, training, num_step):
        """
        A base model forward pass on some data points x. Using the parameters in the weights dictionary. Also requires
        boolean flags indicating whether to reset the running statistics at the end of the run (if at evaluation phase).
        A flag indicating whether this is the training session and an int indicating the current step's number in the
        inner loop.
        :param x: A data batch of shape b, c, h, w
        :param y: A data targets batch of shape b, n_classes
        :param weights: A dictionary containing the weights to pass to the network.
        :param backup_running_statistics: A flag indicating whether to reset the batch norm running statistics to their
         previous values after the run (only for evaluation)
        :param training: A flag indicating whether the current process phase is a training or evaluation.
        :param num_step: An integer indicating the number of the step in the inner loop.
        :return: the crossentropy losses with respect to the given y, the predictions of the base model.
        """
        preds = self.classifier.forward(x=x, params=weights,
                                        training=training,
                                        backup_running_statistics=backup_running_statistics, num_step=num_step)

        loss = F.cross_entropy(input=preds, target=y)

        return loss, preds

    def trainable_parameters(self):
        """
        Returns an iterator over the trainable parameters of the model.
        """
        for param in self.parameters():
            if param.requires_grad:
                yield param

    def train_forward_prop(self, data_batch, epoch):
        """
        Runs an outer loop forward prop using the meta-model and base-model.
        :param data_batch: A data batch containing the support set and the target set input, output pairs.
        :param epoch: The index of the currrent epoch.
        :return: A dictionary of losses for the current step.
        """
        losses, full_losses, full_accs, per_task_target_preds, dists = self.forward(data_batch=data_batch, epoch=epoch,
                                                     use_second_order=self.args.second_order and
                                                                      epoch > self.args.first_order_to_second_order_epoch,
                                                     use_multi_step_loss_optimization=self.args.use_multi_step_loss_optimization,
                                                     num_steps=self.args.number_of_training_steps_per_iter,
                                                     training_phase=True, hook=False)
        return losses, full_losses, full_accs, per_task_target_preds

    def evaluation_forward_prop(self, data_batch, epoch, hook):
        """
        Runs an outer loop evaluation forward prop using the meta-model and base-model.
        :param data_batch: A data batch containing the support set and the target set input, output pairs.
        :param epoch: The index of the currrent epoch.
        :return: A dictionary of losses for the current step.
        """  
        losses, _, full_accs, per_task_target_preds,dists = self.forward(data_batch=data_batch, epoch=epoch, use_second_order=False,
                                                     use_multi_step_loss_optimization=True,
                                                     num_steps=self.args.number_of_evaluation_steps_per_iter,
                                                     training_phase=False,hook=hook)

        return losses, full_accs, per_task_target_preds, dists

    def meta_update(self, loss):
        """
        Applies an outer loop update on the meta-parameters of the model.
        :param loss: The current crossentropy loss.
        """
        self.optimizer.zero_grad()
        loss.backward()
        if 'imagenet' in self.args.dataset_name:
            for name, param in self.classifier.named_parameters():
                if param.requires_grad:
                    param.grad.data.clamp_(-10, 10)  # not sure if this is necessary, more experiments are needed
        self.optimizer.step()
        
    # The following code is adapted from the psuedocode in https://arxiv.org/pdf/1309.1541.pdf (Wang and Perpinan 2013)
    def simplex_proj(self, beta):
        beta_sorted = np.flip(np.sort(beta))
        rho = 1
        for i in range(len(beta)-1):
            j = len(beta) - i
            test = beta_sorted[j-1] + (1 - np.sum(beta_sorted[:j]))/(j)
            if test > 0:
                rho = j
                break

        lam = (1-np.sum(beta_sorted[:rho]))/(rho)
        return np.maximum(beta + lam,0)

    def run_train_iter(self, data_batch, epoch):
        """
        Runs an outer loop update step on the meta-model's parameters.
        :param data_batch: input data batch containing the support set and target set input, output pairs
        :param epoch: the index of the current epoch
        :return: The losses of the ran iteration.
        """
        epoch = int(epoch)
        self.scheduler.step(epoch=epoch)
        if self.current_epoch != epoch:
            self.current_epoch = epoch

        if not self.training:
            self.train()

        x_support_set, x_target_set, y_support_set, y_target_set, selected_task = data_batch

 #       print(selected_task[0])
#        print(x_support_set[0][0,:,:,:,:])

        x_support_set = torch.Tensor(x_support_set).float().to(device=self.device)
        x_target_set = torch.Tensor(x_target_set).float().to(device=self.device)
        y_support_set = torch.Tensor(y_support_set).long().to(device=self.device)
        y_target_set = torch.Tensor(y_target_set).long().to(device=self.device)
        
#         p = torch.Tensor(p).long().to(device=self.device)

        data_batch = (x_support_set, x_target_set, y_support_set, y_target_set,selected_task)
        #print(x_support_set[0])
        losses, full_losses, full_accs, per_task_target_preds = self.train_forward_prop(data_batch=data_batch, epoch=epoch)
        
        loss_report = np.zeros(self.num_tasks)
        loss_counts = np.zeros(self.num_tasks)
        for i, idx in enumerate(selected_task):
            loss_report[idx] += full_losses[i]
            loss_counts[idx] += 1
        for idx in range(self.num_tasks):
            if loss_counts[idx] > 1:
                loss_report[idx] = loss_report[idx]/loss_counts[idx]

    #    print("TASK_train")
     #   print(selected_task)
       # print(loss_report)        

        losses['check_mx']=False
        if losses['mx_loss'] > self.max:
            self.max = losses['mx_loss']
        self.count += 1
        if self.count == 1:
            losses['mx_loss']= self.max
            losses['check_mx'] = True
            self.old_max = self.max
            self.max = 0
            self.count = 0
        else:
            losses['mx_loss'] = self.old_max

        full_losses = torch.stack(full_losses)
        losses['loss'] = torch.mean(full_losses)
        if self.TR_MAML and epoch != self.old_epoch:
            print(self.p)
            self.old_epoch=epoch
        for idx, task_idx in enumerate(selected_task):
            full_losses[idx] = self.p[task_idx]*full_losses[idx]*self.num_tasks

        self.meta_update(torch.mean(full_losses))
        losses['learning_rate'] = self.scheduler.get_lr()[0]

        # update p
        if self.TR_MAML==1:
            tmp = np.zeros(self.num_tasks)
            for idxx, task_idx in enumerate(selected_task):
                tmp[task_idx] += full_losses[idxx].detach().cpu().numpy()
            self.p = self.simplex_proj(self.p + self.lr_p*tmp)
        
        self.optimizer.zero_grad()
        self.zero_grad()

        return losses, per_task_target_preds

    def run_validation_iter(self, data_batch, hook):
        """
        Runs an outer loop evaluation step on the meta-model's parameters.
        :param data_batch: input data batch containing the support set and target set input, output pairs
        :param epoch: the index of the current epoch
        :return: The losses of the ran iteration.
        """

        if self.training:
            self.eval()

        x_support_set, x_target_set, y_support_set, y_target_set, selected_task = data_batch

        x_support_set = torch.Tensor(x_support_set).float().to(device=self.device)
        x_target_set = torch.Tensor(x_target_set).float().to(device=self.device)
        y_support_set = torch.Tensor(y_support_set).long().to(device=self.device)
        y_target_set = torch.Tensor(y_target_set).long().to(device=self.device)

        #print(self.p)

        data_batch = (x_support_set, x_target_set, y_support_set, y_target_set,selected_task)

        losses, full_accs, per_task_target_preds, dists = self.evaluation_forward_prop(data_batch=data_batch, epoch=self.current_epoch, hook=hook)

        if losses['mx_loss'] > self.max:
            self.max = losses['mx_loss']
        self.count += 1
        if self.count == 5:
            losses['mx_loss'] = self.max
            self.old_val_max = self.max
            self.max = 0
            self.count = 0
        else:
            losses['mx_loss'] = self.old_val_max


        # losses['loss'].backward() # uncomment if you get the weird memory error
        # self.zero_grad()
        # self.optimizer.zero_grad()

        return losses, full_accs, per_task_target_preds, dists

    def save_model(self, model_save_dir, state):
        """
        Save the network parameter state and experiment state dictionary.
        :param model_save_dir: The directory to store the state at.
        :param state: The state containing the experiment state and the network. It's in the form of a dictionary
        object.
        """
        state['network'] = self.state_dict()
        torch.save(state, f=model_save_dir)

    def load_model(self, model_save_dir, model_name, model_idx):
        """
        Load checkpoint and return the state dictionary containing the network state params and experiment state.
        :param model_save_dir: The directory from which to load the files.
        :param model_name: The model_name to be loaded from the direcotry.
        :param model_idx: The index of the model (i.e. epoch number or 'latest' for the latest saved model of the current
        experiment)
        :return: A dictionary containing the experiment state and the saved model parameters.
        """
        filepath = os.path.join(model_save_dir, "{}_{}".format(model_name, model_idx))
        state = torch.load(filepath)
        state_dict_loaded = state['network']
        self.load_state_dict(state_dict=state_dict_loaded)
        return state

