# JephHinter Workflow Remote Execution

This document explains how to run the JephHinter workflow both locally and remotely using the updated codebase.

## Overview

The JephHinter workflow consists of three main steps:
1. **Step 1**: Run experiments without hints
2. **Step 2**: Generate hints from traces using JephHinter
3. **Step 3**: Run experiments with hints enabled

The workflow can now be executed either locally (interactive) or remotely as a job.

## Configuration

### JephHinterWorkflowConfig

The main configuration class has been updated to include an `interactive_job` parameter:

```python
from agentlab_configs import JephHinterWorkflowConfig

config = JephHinterWorkflowConfig(
    exp_root="/path/to/experiments",
    interactive_job=True,  # True for local execution, False for remote
    openai_key="your-openai-key",
    anthropic_key="your-anthropic-key",
    # ... other parameters
)
```

### Key Parameters

- `interactive_job`: Controls execution mode
  - `True`: Run locally (interactive)
  - `False`: Launch as remote job
- `exp_root`: Base directory for experiment results
- `openai_key`: OpenAI API key for LLM access
- `anthropic_key`: Anthropic API key for LLM access
- `snow_pwd`, `snow_url`, `snow_username`: ServiceNow credentials
- `miniwob_url`: MiniWoB URL for experiments

## Usage

### Local Execution

To run the workflow locally:

```python
from agentlab_configs import JephHinterWorkflowConfig
from launch_jephhinter import run_workflow_locally

config = JephHinterWorkflowConfig(
    exp_root="/home/toolkit/AgentLab/my_experiments",
    interactive_job=True,  # Run locally
    openai_key="your-openai-key",
    anthropic_key="your-anthropic-key",
)

run_workflow_locally(config)
```

### Remote Execution

To run the workflow as a remote job:

```python
from agentlab_configs import JephHinterWorkflowConfig
from launch_jephhinter import run_workflow_remotely

config = JephHinterWorkflowConfig(
    exp_root="/home/toolkit/AgentLab/my_experiments",
    interactive_job=False,  # Run remotely
    openai_key="your-openai-key",
    anthropic_key="your-anthropic-key",
)

job_id = run_workflow_remotely(config)
print(f"Job launched with ID: {job_id}")
print(f"Follow progress: eai job logs -f {job_id}")
```

### Command Line Usage

You can also run the workflow directly from the command line:

```bash
# Local execution
python launch_jephhinter.py

# The script will automatically check the interactive_job parameter
# in the JephHinterWorkflowConfig and run accordingly
```

## Remote Job Details

### Resource Allocation

Remote jobs are launched with the following default resources:
- **GPU**: 1 GPU with 80GB memory
- **CPU**: 4 cores
- **Memory**: 64GB RAM
- **Max Runtime**: 4 days

You can customize these in the `run_workflow_remotely` function:

```python
job_id = run_workflow_remotely(
    config,
    n_gpu=2,        # Number of GPUs
    gpu_mem=80,     # GPU memory in GB
    n_cpu=8,        # Number of CPUs
    mem=128,        # Memory in GB
    max_run_time=7 * 24 * 60 * 60,  # 7 days
)
```

### Job Monitoring

Once a remote job is launched, you can monitor it using:

```bash
# View job logs
eai job logs -f <job_id>

# Check job status
eai job info <job_id>

# List all your jobs
eai job ls --account <your-account>
```

### Job Output

The remote job will:
1. Create a temporary script in `/home/toolkit/tmp/`
2. Execute the complete JephHinter workflow
3. Save results to the specified `exp_root` directory
4. Generate logs that can be viewed with `eai job logs`

## File Structure

After execution, you'll have the following structure:

```
exp_root/
├── hint_db_updated.csv          # Generated hints database
├── [experiment files]           # Step 1 results (no hints)
└── -with_jeph/
    └── [experiment files]       # Step 3 results (with hints)
```

## Error Handling

### Local Execution Errors

- Check that all required environment variables are set
- Verify API keys are valid
- Ensure sufficient disk space for experiment results

### Remote Execution Errors

- Check job logs: `eai job logs -f <job_id>`
- Verify job status: `eai job info <job_id>`
- Common issues:
  - Insufficient resources
  - Invalid API keys
  - Network connectivity issues
  - Disk space limitations

## Testing

Use the provided test script to verify functionality:

```bash
python test_remote_jephhinter.py
```

This will test configuration creation and demonstrate both local and remote execution patterns.

## Dependencies

The remote execution functionality requires:
- `finetuning` package (for job launching utilities)
- `toolkit_utils` package (for server management)
- Valid EAI account and credentials
- Proper environment setup

## Troubleshooting

### Import Errors

If you encounter import errors for remote job utilities:

```python
try:
    from utils.remote_job_utils import launch_jephhinter_workflow_remote
    REMOTE_JOB_AVAILABLE = True
except ImportError:
    REMOTE_JOB_AVAILABLE = False
    print("Warning: Remote job utilities not available. Will run locally only.")
```

### Configuration Issues

- Ensure all required API keys are set
- Verify ServiceNow credentials if using WorkArena benchmarks
- Check that experiment paths are writable

### Resource Issues

- Monitor job resource usage
- Adjust resource allocation if needed
- Consider breaking large experiments into smaller chunks

## Examples

### Complete Example

```python
#!/usr/bin/env python3
"""
Complete example of running JephHinter workflow remotely.
"""

from agentlab_configs import JephHinterWorkflowConfig
from launch_jephhinter import run_workflow_remotely

def main():
    # Configure the workflow
    config = JephHinterWorkflowConfig(
        exp_root="/home/toolkit/AgentLab/jephhinter_experiment_001",
        interactive_job=False,  # Run remotely
        openai_key="sk-...",  # Your OpenAI key
        anthropic_key="sk-ant-...",  # Your Anthropic key
        snow_pwd="your-snow-password",
        snow_url="https://your-snow-instance.service-now.com",
        snow_username="your-username",
    )
    
    # Launch remote job
    print("Launching JephHinter workflow as remote job...")
    job_id = run_workflow_remotely(config)
    
    if job_id:
        print(f"✅ Job launched successfully!")
        print(f"Job ID: {job_id}")
        print(f"Monitor progress: eai job logs -f {job_id}")
        print(f"Check status: eai job info {job_id}")
    else:
        print("❌ Failed to launch job")

if __name__ == "__main__":
    main()
```

This setup provides a complete solution for running the JephHinter workflow both locally and remotely, with proper error handling and monitoring capabilities. 