"""
Configuration dataclasses for AgentLab JephHinter workflow.
Centralized configuration management for all components.
"""

from dataclasses import dataclass, field
from typing import Literal

from agentlab.agents.agent_args import AgentArgs
from agentlab.agents.tool_use_agent import AGENT_CONFIG, CLAUDE_SONNET_37
from agentlab.experiments.exp_utils import RESULTS_DIR
from agentlab.llm.base_api import BaseModelArgs

DEFAULT_EXP_PATH = RESULTS_DIR.as_posix()


@dataclass
class HintPromptConfig:
    """Configuration for hint prompt generation."""

    exclude_html: bool = True
    exclude_axtree: bool = True
    exclude_actions: bool = False
    exclude_think: bool = False
    exclude_reward: bool = False
    n_traces_to_hinter: int = 2
    n_hints_per_task: int = 5
    n_steps_for_hinting: int = 3
    use_step_zoom: bool = False
    step_wise_hinting: bool = False 

@dataclass
class JephHinterConfig:
    """Configuration for JephHinter agent."""

    traces_folder: str = ""
    hint_prompt_config: HintPromptConfig = field(default_factory=lambda: HintPromptConfig())
    max_traces: int = 100
    hint_db_path: str = "hint_db.csv"
    agent_name: str = "JephHinter"
    user_name: str = "auto"
    source: str = "jeph_hinter"
    domain_name: str = ""
    save_chat_messages: bool = True
    chat_output_dir: str = "chat_messages"
    version: str = "v1"

@dataclass
class AutoGuideConfig_workarena_l1:
    """Configuration for AutoGuide agent."""

    traces_folder: str = ""
    hint_prompt_config: HintPromptConfig = field(default_factory=lambda: HintPromptConfig(n_traces_to_hinter=2, exclude_html=True, exclude_axtree=True))
    max_traces: int = 100
    hint_db_path: str = "hint_db.csv"
    agent_name: str = "AutoGuide"
    user_name: str = "auto"
    source: str = "autoguide"
    domain_name: str = ""
    save_chat_messages: bool = True
    chat_output_dir: str = "chat_messages"
    version: str = "v1"

@dataclass
class AutoGuideConfig_miniwob:
    """Configuration for AutoGuide agent."""

    traces_folder: str = ""
    hint_prompt_config: HintPromptConfig = field(default_factory=lambda: HintPromptConfig(n_traces_to_hinter=2, exclude_html=False, exclude_axtree=True))
    max_traces: int = 100
    hint_db_path: str = "hint_db.csv"
    agent_name: str = "AutoGuide"
    user_name: str = "auto"
    source: str = "autoguide"
    domain_name: str = ""
    save_chat_messages: bool = True
    chat_output_dir: str = "chat_messages"
    version: str = "v1"

@dataclass
class AgentLabRunConfig:
    """Configuration for AgentLab experiment runs."""

    exp_root: str = DEFAULT_EXP_PATH
    use_task_hint: bool = False
    hint_db_path: str = "hint_db.csv"
    hint_mode: Literal["direct", "llm", "emb"] = "direct"
    reproducibility_mode: bool = False
    relaunch: bool = False
    n_jobs: int = 5
    n_repeats: int = 5
    n_relaunch: int = 5
    backend: Literal["ray", "sequential"] = "ray"
    max_steps: int = 10
    benchmark_name: Literal["miniwob", "workarena_l1"] = "miniwob"
    agent_name: str = "agent"
    agent_args: AgentArgs = field(default_factory=lambda: AGENT_CONFIG)


@dataclass
class HintsMinerConfig:
    """Configuration for hint mining process."""

    root_dir: str = DEFAULT_EXP_PATH
    output_path: str = "hint_db.csv"
    model_args: BaseModelArgs = field(default_factory=lambda: CLAUDE_SONNET_37)
    hinter_config: JephHinterConfig = field(default_factory=lambda: JephHinterConfig())


@dataclass
class JephHinterWorkflowConfig:
    """Configuration for the complete JephHinter workflow."""

    exp_root: str
    interactive_job: bool = True
    eval_config: AgentLabRunConfig = field(
        default_factory=lambda: AgentLabRunConfig(exp_root=DEFAULT_EXP_PATH)
    )
    hinter_config: JephHinterConfig = field(default_factory=lambda: JephHinterConfig())
    hinter_models: list[BaseModelArgs] = field(default_factory=lambda: [CLAUDE_SONNET_37])
