#!/usr/bin/env python3
"""
Python launcher for JephHinter workflow
This script runs the complete workflow:
1. Run experiments without hints
2. Generate hints from traces using JephHinter
3. Run experiments with hints enabled
"""

import json
import logging
import os
import sys
from datetime import datetime
from pathlib import Path

from agentlab.agents.generic_agent import FLAGS_GPT_4o, GenericAgentArgs
from agentlab.agents.tool_use_agent import (
    DEFAULT_PROMPT_CONFIG,
    GPT_5,
    ToolUseAgentArgs,
)

from jephhinter.emb_server import encode_hints_db

# Add the project root to the Python path
project_root = Path(__file__).parent.parent
sys.path.insert(0, str(project_root))
print(sys.path)

from copy import deepcopy
from dataclasses import asdict

from dotenv import load_dotenv

from jephhinter.agentlab_run import AgentLabRun
from jephhinter.configs import (
    RESULTS_DIR,
    AgentLabRunConfig,
    HintsMinerConfig,
    JephHinterWorkflowConfig,
)
from jephhinter.jeph_hinter import HintsMiner

logger = logging.getLogger(__name__)
logger.setLevel(logging.INFO)
load_dotenv()


# Import remote job utilities
try:
    # Try relative import first
    from .utils.remote_job_utils import launch_jephhinter_workflow_remote

    REMOTE_JOB_AVAILABLE = True
except ImportError:
    try:
        # Try absolute import
        from jephhinter.utils.remote_job_utils import launch_jephhinter_workflow_remote

        REMOTE_JOB_AVAILABLE = True
    except ImportError:
        REMOTE_JOB_AVAILABLE = False
        logger.info("Warning: Remote job utilities not available. Will run locally only.")


class JephHinterWorkflow:
    """
    Complete workflow for running JephHinter experiments.
    """

    def __init__(self, config: JephHinterWorkflowConfig):
        """
        Initialize the workflow.

        Args:
            config: Configuration object containing all workflow parameters
            agentlab_config: AgentLab configuration
            jephhinter_model: JephHinter model configuration
        """
        self.config = config
        exp_root = Path(config.exp_root)
        if not exp_root.is_absolute():
            exp_root = RESULTS_DIR / exp_root

        self.exp_root_no_hint = f"{exp_root}"
        self.exp_roots_with_hint = []
        self.db_paths = []
        for hinter_model in config.hinter_models:
            path = f"{exp_root}_with_hints_{config.hinter_config.source}_{config.hinter_config.version}_{hinter_model.model_name}"
            self.exp_roots_with_hint.append(path)
            self.db_paths.append(os.path.join(path, "hint_db_updated.csv"))

    def _save_config_to_folder(self, folder_path: str, config_name: str = "config"):
        """Save configuration to a JSON file in the specified folder."""

        # Create the folder if it doesn't exist
        os.makedirs(folder_path, exist_ok=True)

        # Save workflow config
        workflow_config_path = os.path.join(folder_path, f"{config_name}_workflow.json")
        workflow_dict = asdict(self.config)
        with open(workflow_config_path, "w") as f:
            json.dump(workflow_dict, f, indent=2, default=str)

        logger.info(f"✅ Configuration saved to {folder_path}")

    def _log_header(self, step_name: str):
        """Print a formatted header for each step."""
        logger.info(f"\n{'=' * 50}")
        logger.info(f"=== {step_name} ===")
        logger.info(f"{'=' * 50}")

    def _log_config(self):
        """Print the current configuration."""
        logger.info("Configuration:")
        logger.info(f"  No-hint experiments: {self.exp_root_no_hint}")
        logger.info(f"  With-hint experiments: {self.exp_roots_with_hint}")
        logger.info(f"  Hint databases: {self.db_paths}")
        logger.info(f"  Interactive job: {self.config.interactive_job}\n")
        logger.info(
            f"  Hinter models: {[model.model_name for model in self.config.hinter_models]}\n"
        )

    def _create_directories(self):
        """Create necessary directories."""
        logger.info("Creating necessary directories...")
        os.makedirs(self.exp_root_no_hint, exist_ok=True)
        for path in self.exp_roots_with_hint:
            os.makedirs(path, exist_ok=True)
        logger.info("Directories created successfully!\n")

    def run_without_hints(self, eval_config: AgentLabRunConfig):
        """Run AgentLab experiments without hints.

        Args:
            agentlab_config: Configuration for AgentLab experiments

        Returns:
            bool: True if successful, False otherwise
        """

        self._log_header("Step 1: Running experiments without hints")
        logger.info(
            f"Running experiments with exp-root={self.exp_root_no_hint} (hints disabled by default)"
        )

        # Override with workflow-specific settings
        agentlab_config = deepcopy(eval_config)
        agentlab_config.exp_root = self.exp_root_no_hint
        agentlab_config.use_task_hint = False

        try:
            AgentLabRun(agentlab_config).run()
        except Exception as e:
            logger.error(f"Error occurred while running experiment without hints: {e}")
            return False
        logger.info("Step 1 complete!")
        return True

    def run_with_hints(self, eval_config: AgentLabRunConfig):
        """Run AgentLab experiments with hints.

        Args:
            agentlab_config: Configuration for AgentLab experiments

        Returns:
            bool: True if successful, False otherwise
        """
        self._log_header("Step 3: Running experiments with hints")
        for i, exp_path in enumerate(self.exp_roots_with_hint):
            db_path = self.db_paths[i]
            logger.info(f"Running experiments with exp-root={exp_path} (hints enabled)")

            # Check if hint database exists
            if not os.path.exists(db_path):
                logger.info(f"Error: Hint database not found at {db_path}")
                logger.info("Please ensure hint generation completed successfully.")
                return False

            agentlab_config = deepcopy(eval_config)
            agentlab_config.exp_root = exp_path
            agentlab_config.use_task_hint = True
            agentlab_config.hint_db_path = db_path

            try:
                AgentLabRun(agentlab_config).run()
            except Exception as e:
                logger.error(f"Error occurred while running experiment with hints: {e}")
                return False

        logger.info("Step 3 complete!")
        return True

    def generate_hints(self):
        """Step 2: Generate hints using JephHinter."""
        self._log_header("Hint Generation: Generating hints from traces")
        logger.info(f"Running JephHinter with root-dir={self.exp_root_no_hint}")

        for i, hinter_model in enumerate(self.config.hinter_models):
            db_path = self.db_paths[i]
            logger.info(f"Using JephHinter model: {hinter_model.model_name}")

            try:
                HintsMiner(
                    HintsMinerConfig(
                        root_dir=self.exp_root_no_hint,
                        output_path=db_path,
                        model_args=hinter_model,
                        hinter_config=self.config.hinter_config,
                    )
                ).run()
                logger.info(f"Hint database created at: {db_path}")
                embs_path = encode_hints_db(db_path)
                logger.info(f"Hint embeddings saved to: {embs_path}")

            except Exception as e:
                logger.exception(f"Error occurred while generating hints: {e}")
                return False
        logger.info("Hint Generation complete!")
        return True

    def run_complete_workflow(self):
        """Run the complete JephHinter workflow."""
        self._log_header("JephHinter Workflow")
        self._log_config()
        self._create_directories()

        # Step 1. Data collection: Run experiments without hints
        data_collection_success = self.run_without_hints(self.config.eval_config)
        if not data_collection_success:
            logger.info("❌ Data collection failed. Aborting workflow.")
            return False

        # Step 2. Hint generation: Generate hints using JephHinter
        db_ready = self.generate_hints()
        if not db_ready:
            logger.info("❌ Hint generation failed. Aborting workflow.")
            return False

        # Step 3. Evaluation: Run experiments with hints enabled
        evaluation_success = self.run_with_hints(self.config.eval_config)

        # Save configuration to the with-hint experiments folder
        if evaluation_success:
            for exp_path in self.exp_roots_with_hint:
                self._save_config_to_folder(exp_path, "jephhinter_config")

        # Summary
        self._log_header("JephHinter Workflow Complete")
        logger.info(f"Results without hints: {self.exp_root_no_hint}")
        logger.info(f"Results with hints: {self.exp_roots_with_hint}")
        logger.info(f"Hinter models {[model.model_name for model in self.config.hinter_models]}")
        logger.info(f"Hint databases: {self.db_paths}")

        if data_collection_success and evaluation_success:
            logger.info("\n✅ All steps completed successfully!")
            return True
        else:
            logger.info("\n❌ Some steps failed. Please check the output above.")
            return False

    def run_data_collection_only(self, with_hint: bool = False):
        """Run only data collection: Collect no-hint experiments (run once per agent)."""
        self._log_header("Data Collection Only: Collecting Data Experiments")
        self._log_config()
        self._create_directories()

        if with_hint:
            # Data collection: Run experiments with hints
            data_collection_success = self.run_with_hints(self.config.eval_config)
        else:
        # Data collection: Run experiments without hints
            data_collection_success = self.run_without_hints(self.config.eval_config)


        if data_collection_success:
            logger.info("\n✅ Data collection completed successfully!")
            logger.info(f"Experiments saved to: {self.exp_root_no_hint}")
            return True
        else:
            logger.info("\n❌ Data collection failed!")
            return False

    def run_hint_generation_and_evaluation(self, agentlab_config: AgentLabRunConfig):
        """Run hint generation and evaluation: Generate hints and run with-hint experiments (run for each agent-model combination)."""
        self._log_header(
            "Hint Generation and Evaluation: Generate Hints and Run With-Hint Experiments"
        )
        self._log_config()
        self._create_directories()  # Create necessary directories including exp_root_with_hint

        # Hint generation: Generate hints using JephHinter
        db_ready = self.generate_hints()
        if not db_ready:
            logger.info("❌ Hint generation failed. Aborting workflow.")
            return False

        # Evaluation: Run experiments with hints enabled
        evaluation_success = self.run_with_hints(agentlab_config)

        # Save configuration to the with-hint experiments folder
        if evaluation_success:
            for exp_path in self.exp_roots_with_hint:
                self._save_config_to_folder(exp_path, "jephhinter_config")

        # Summary
        self._log_header("Hint Generation and Evaluation Complete")
        logger.info(f"Results with hints: {self.exp_roots_with_hint}")
        logger.info(f"Hint databases: {self.db_paths}")

        if evaluation_success:
            logger.info("\n✅ Hint generation and evaluation completed successfully!")
            return True
        else:
            logger.info("\n❌ Hint generation and evaluation failed!")
            return False

    def run_hint_generation_only(self):
        """Run only hint generation: Generate hints using JephHinter."""
        self._log_header("Hint Generation Only: Generating hints from traces")
        self._log_config()
        self._create_directories()

        # Hint generation: Generate hints using JephHinter
        db_ready = self.generate_hints()
        return db_ready


def run_data_collection_only_locally(config: JephHinterWorkflowConfig):
    """Run only data collection locally: Collect no-hint experiments (run once per agent)."""
    workflow = JephHinterWorkflow(config)
    success = workflow.run_data_collection_only()
    if success:
        logger.info("✅ Data collection completed successfully!")
    else:
        logger.info("❌ Data collection failed!")
    return success


def run_hint_generation_and_evaluation_locally(config: JephHinterWorkflowConfig):
    """Run hint generation and evaluation locally: Generate hints and run with-hint experiments."""
    workflow = JephHinterWorkflow(config)
    success = workflow.run_hint_generation_and_evaluation(config.eval_config)
    if success:
        logger.info("✅ Hint generation and evaluation completed successfully!")
    else:
        logger.info("❌ Hint generation and evaluation failed!")
    return success


def run_data_collection_only_remotely(config: JephHinterWorkflowConfig, with_hint: bool = False):
    """Run only data collection remotely: Collect no-hint experiments (run once per agent)."""
    if not REMOTE_JOB_AVAILABLE:
        logger.info("Error: Remote job utilities not available. Cannot run remotely.")
        return None

    env_vars = {
        "SNOW_INSTANCE_PWD": os.environ.get("SNOW_INSTANCE_PWD"),
        "SNOW_INSTANCE_URL": os.environ.get("SNOW_INSTANCE_URL"),
        "SNOW_INSTANCE_UNAME": os.environ.get("SNOW_INSTANCE_UNAME"),
        "OPENAI_API_KEY": os.environ.get("OPENAI_API_KEY"),
        "ANTHROPIC_API_KEY": os.environ.get("ANTHROPIC_API_KEY"),
        "OPENROUTER_API_KEY": os.environ.get("OPENROUTER_API_KEY"),
        "MINIWOB_URL": os.environ.get("MINIWOB_URL"),
    }
    workflow_mode = "data_collection_only"

    # Launch remote job
    job_id = launch_jephhinter_workflow_remote(  # type: ignore
        config,
        env_vars,
        workflow_mode,
        n_gpu=1,
        gpu_mem=80,
        n_cpu=4,
        mem=64,
        max_run_time=4 * 24 * 60 * 60,  # 4 days
        with_hint=with_hint,
    )

    logger.info(f"✅ Data collection launched as remote job with ID: {job_id}")
    logger.info(f"Follow the job using: eai job logs -f {job_id}")
    return job_id


def run_hint_generation_and_evaluation_remotely(config: JephHinterWorkflowConfig):
    """Run hint generation and evaluation remotely: Generate hints and run with-hint experiments."""
    if not REMOTE_JOB_AVAILABLE:
        logger.info("Error: Remote job utilities not available. Cannot run remotely.")
        return None

    env_vars = {
        "SNOW_INSTANCE_PWD": os.environ.get("SNOW_INSTANCE_PWD"),
        "SNOW_INSTANCE_URL": os.environ.get("SNOW_INSTANCE_URL"),
        "SNOW_INSTANCE_UNAME": os.environ.get("SNOW_INSTANCE_UNAME"),
        "OPENAI_API_KEY": os.environ.get("OPENAI_API_KEY"),
        "ANTHROPIC_API_KEY": os.environ.get("ANTHROPIC_API_KEY"),
        "OPENROUTER_API_KEY": os.environ.get("OPENROUTER_API_KEY"),
        "MINIWOB_URL": os.environ.get("MINIWOB_URL"),
    }
    workflow_mode = (
        "hint_generation_and_evaluation"  # Indicate this is hint generation and evaluation only
    )
    # Launch remote job
    job_id = launch_jephhinter_workflow_remote(  # type: ignore
        config,
        env_vars,
        workflow_mode,
        n_gpu=1,
        gpu_mem=80,
        n_cpu=4,
        mem=64,
        max_run_time=4 * 24 * 60 * 60,  # 4 days
        with_hint=True,
    )

    logger.info(f"✅ Hint generation and evaluation launched as remote job with ID: {job_id}")
    logger.info(f"Follow the job using: eai job logs -f {job_id}")
    return job_id


def run_workflow_locally(config: JephHinterWorkflowConfig):
    """Run the workflow locally."""
    logger.info("Running JephHinter workflow locally")
    success = JephHinterWorkflow(config).run_complete_workflow()
    if success:
        logger.info("✅ Workflow completed successfully!")
    else:
        logger.info("❌ Workflow failed!")

    return success


def run_workflow_remotely(config: JephHinterWorkflowConfig):
    """Run the workflow remotely as a job."""
    if not REMOTE_JOB_AVAILABLE:
        logger.info("Error: Remote job utilities not available. Cannot run remotely.")
        return
    logger.info("Launching JephHinter workflow as remote job...")

    env_vars = {
        "SNOW_INSTANCE_PWD": os.environ.get("SNOW_INSTANCE_PWD"),
        "SNOW_INSTANCE_URL": os.environ.get("SNOW_INSTANCE_URL"),
        "SNOW_INSTANCE_UNAME": os.environ.get("SNOW_INSTANCE_UNAME"),
        "OPENAI_API_KEY": os.environ.get("OPENAI_API_KEY"),
        "ANTHROPIC_API_KEY": os.environ.get("ANTHROPIC_API_KEY"),
        "OPENROUTER_API_KEY": os.environ.get("OPENROUTER_API_KEY"),
        "MINIWOB_URL": os.environ.get("MINIWOB_URL"),
    }
    workflow_mode = "complete"

    # Launch remote job
    job_id = launch_jephhinter_workflow_remote(  # type: ignore
        config,
        env_vars,
        workflow_mode,
        n_gpu=1,
        gpu_mem=80,
        n_cpu=4,
        mem=64,
        max_run_time=4 * 24 * 60 * 60,  # 4 days
    )

    logger.info(f"✅ JephHinter workflow launched as remote job with ID: {job_id}")
    logger.info(f"Follow the job using: eai job logs -f {job_id}")
    return job_id


def make_generic_like_agent(model_args):
    GENERIC_AGENT_LIKE = deepcopy(DEFAULT_PROMPT_CONFIG)
    GENERIC_AGENT_LIKE.keep_last_n_obs = 1
    GENERIC_AGENT_LIKE.obs.use_screenshot = False
    GENERIC_AGENT_LIKE.action_subsets = ("bid",)
    return GenericAgentArgs(
        chat_model_args=model_args,
        flags=FLAGS_GPT_4o,
    )


if __name__ == "__main__":
    # Example usage
    timestamp = datetime.now().strftime("%Y-%m-%d_%H-%M-%S")
    wf_config = JephHinterWorkflowConfig(
        exp_root=f"{timestamp}-retrievetest1",
        hinter_models=[GPT_5],
    )
    wf_config.eval_config.n_jobs = 10
    wf_config.eval_config.n_repeats = 1
    wf_config.eval_config.backend = "ray"
    wf_config.eval_config.benchmark_name = "miniwob"  # "workarena_l1"

    tooluse = ToolUseAgentArgs(model_args=GPT_5, config=DEFAULT_PROMPT_CONFIG)
    tooluse.config.action_subsets = ("bid",)

    wf_config.eval_config.agent_args = tooluse

    if wf_config.interactive_job:
        JephHinterWorkflow(wf_config).run_complete_workflow()
    else:
        run_workflow_remotely(wf_config)
