import logging
import os
import pickle
from datetime import datetime

# Import only what we need to avoid circular imports
try:
    from jephhinter.toolkit_utils.toolkit_configs import PERSONAL_ACCOUNT_NAME_SUFFIX
except ImportError:
    # Fallback if the import fails
    PERSONAL_ACCOUNT_NAME_SUFFIX = "ui_assist"  # Default fallback

from jephhinter.configs import JephHinterWorkflowConfig


def launch_jephhinter_workflow_remote(
    config: JephHinterWorkflowConfig,
    env_vars: dict,
    workflow_mode: str,
    n_gpu: int = 1,
    gpu_mem: int = 80,
    n_cpu: int = 4,
    mem: int = 64,
    max_run_time: int = 4 * 24 * 60 * 60,
    with_hint: bool = False,
):
    """
    Launches a remote job to run the complete JephHinter workflow.

    Args:
        config_dict: Dictionary containing the JephHinterWorkflowConfig parameters
        n_gpu: Number of GPUs to allocate
        gpu_mem: GPU memory in GB
        n_cpu: Number of CPUs to allocate
        mem: Memory in GB
        max_run_time: Maximum run time in seconds

    Returns:
        job_id: The ID of the launched job
    """
    # Import the launch_jephhinter function from toolkit_servers
    from jephhinter.toolkit_utils.toolkit_servers import launch_jephhinter

    # Create a temporary directory to store the script
    os.makedirs("/home/toolkit/tmp", exist_ok=True)

    job_name = PERSONAL_ACCOUNT_NAME_SUFFIX
    job_name = f"{job_name}_jephhinter_workflow_{datetime.now().strftime('%Y%m%d_%H%M%S')}"

    # Pickle the config dictionary
    config_pkl_file = f"/home/toolkit/tmp/{job_name}_config.pkl"
    with open(config_pkl_file, "wb") as f:
        pickle.dump(config, f)

    # Create the Python script to run the workflow
    script_file = f"/home/toolkit/tmp/{job_name}_script.py"
    with open(script_file, "w") as f:
        f.write(
            f"""#!/usr/bin/env python3
import sys
import os
from pathlib import Path

# Add the current project paths
sys.path.insert(0, "/home/toolkit/ui-copilot-aug16")
sys.path.insert(0, "/home/toolkit/ui-copilot-aug16/jephhinter")

# Set environment variables
{chr(10).join([f'os.environ["{key}"] = "{value}"' for key, value in env_vars.items()])}

# Set AGENTLAB_EXP_ROOT environment variable
os.environ["AGENTLAB_EXP_ROOT"] = "/mnt/ui_assist/data_rw/jephhinter_iclr/workarena-l1_with_1_golden_traces/"

from jephhinter.jephhinter_workflow import JephHinterWorkflow
from jephhinter.configs import JephHinterWorkflowConfig, AgentLabRunConfig
import pickle

# Create config from pickle file
with open("{config_pkl_file}", "rb") as f:
    config = pickle.load(f)

# Get workflow mode
workflow_mode = "{workflow_mode}"

# Run the workflow based on mode
workflow = JephHinterWorkflow(config)

if workflow_mode == "data_collection_only":
    workflow.run_data_collection_only(with_hint={with_hint})
elif workflow_mode == "hint_generation_and_evaluation":
    workflow.run_hint_generation_and_evaluation(config.eval_config)
else:
    # Default: run complete workflow
    workflow.run_complete_workflow()
"""
        )

    # Create the bash script that will be executed
    bash_script = f"/home/toolkit/tmp/{job_name}.sh"
    with open(bash_script, "w") as f:
        f.write(
            f"""#!/bin/bash
cd /home/toolkit/AgentLab

# Use the agentlab conda environment's Python directly
AGENTLAB_PYTHON="/home/toolkit/.conda/envs/agentlab2/bin/python"

# Verify the Python executable exists
if [ ! -f "$AGENTLAB_PYTHON" ]; then
    echo "Error: agentlab conda environment not found at $AGENTLAB_PYTHON"
    echo "Available conda environments:"
    ls -la /home/toolkit/.conda/envs/
    exit 1
fi

echo "Using Python: $AGENTLAB_PYTHON"
echo "Python version: $($AGENTLAB_PYTHON --version)"

# Run the workflow script with the agentlab Python
$AGENTLAB_PYTHON {script_file}
"""
        )
    os.chmod(bash_script, 0o755)

    # Launch the job using the launch_jephhinter function
    # This function takes: job_name, script_path, n_gpu, gpu_mem, n_cpu, mem, max_run_time
    job_id = launch_jephhinter(
        job_name=job_name,
        script_path=bash_script,
        n_gpu=n_gpu,
        gpu_mem=gpu_mem,
        n_cpu=n_cpu,
        mem=mem,
        max_run_time=max_run_time,
    )

    logging.info(f"Launched JephHinter workflow job with ID: {job_id}")
    logging.info(
        f"Job will run the complete workflow with exp_root: {config.exp_root}"
    )
    logging.info("Using conda environment: agentlab")
    logging.info(f"Follow it using the command `eai job logs -f {job_id}`")

    return job_id
