import fnmatch
import itertools
import os
import string

import matplotlib.pyplot as plt
from cycler import cycler

import args_parser


def find(pattern, path):
    result = []
    for root, dirs, files in os.walk(path):
        for name in files:
            if fnmatch.fnmatch(name, pattern):
                result.append(os.path.join(root, name))
    return result


def plot():

    """ Plot figures """
    plt.rcParams.update({
        "text.usetex": True,
        "font.family": "sans-serif",
        "font.size": 24,
        "font.sans-serif": ["Helvetica"],
    })

    i = 1
    skip_n = 1

    games = ['SIS', 'SIR', 'RS', ]
    variants = ["fpi", "omd", ]

    for game in games:
        clist = itertools.cycle(cycler(color='rbkgcmy'))
        linestyle_cycler = itertools.cycle(cycler('linestyle', ['-', '--', ':', '-.']))
        subplot = plt.subplot(1, len(games), i)
        subplot.annotate('(' + string.ascii_lowercase[i - 1] + ')',
                         (0, 0),
                         xytext=(10, +32),
                         xycoords='axes fraction',
                         textcoords='offset points',
                         fontweight='bold',
                         color='black',
                         alpha=0.7,
                         backgroundcolor='white',
                         ha='left', va='top')
        # subplot.text(-0.01, 1.06, '(' + string.ascii_lowercase[i - 1] + ')', transform=subplot.transAxes, weight='bold')
        i += 1

        for variant in variants:
            plot_vals = []

            config = args_parser.generate_config_from_kw(game=game, variant=variant, fp_iterations=5000, temperature=50)
            files = find('stdout', config['exp_dir'])

            with open(max(files, key=os.path.getctime), 'r') as fi:
                fi_lines = fi.readlines()
                for line in fi_lines[:]:
                    fields = line.split(" ")
                    plot_vals.append(float(fields[3][:-1]))

            color = clist.__next__()['color']
            linestyle = linestyle_cycler.__next__()['linestyle']
            subplot.loglog(range(len(plot_vals))[::skip_n], plot_vals[::skip_n], linestyle, color=color,
                         label="FPI" if variant == 'fpi' else "HOMD", alpha=0.5, linewidth=2)

        if i%3==0:
            lgd1 = plt.legend(loc='lower center', ncol=1, fontsize="20")
        # plt.title(game + " " + variant)
        plt.grid('on')
        plt.xlabel(r'Iterations $n$', fontsize=22)
        if i%3==2:
            plt.ylabel(r'Exploitability $\mathcal E$', fontsize=22)
        plt.xlim([0, len(plot_vals)-1])
        plt.xscale('symlog')
        if i%3==1:
            plt.yscale('symlog', linthresh=1e-3)

    """ Finalize plot """
    plt.gcf().set_size_inches(13, 3.5)
    plt.tight_layout(w_pad=0.0)
    plt.savefig(f'./figures/exploitability.pdf', bbox_extra_artists=(lgd1,), bbox_inches='tight', transparent=True, pad_inches=0.1)
    plt.savefig(f'./figures/exploitability.png', bbox_extra_artists=(lgd1,), bbox_inches='tight', transparent=True, pad_inches=0.1)
    plt.show()


if __name__ == '__main__':
    plot()
