# Modeling Irregular Time Series with Continuous Recurrent Units (CRUs)
# Copyright (c) 2022 Robert Bosch GmbH
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Affero General Public License as published
# by the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU Affero General Public License for more details.
#
# You should have received a copy of the GNU Affero General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
#
# This source code is derived from Pytorch RKN Implementation (https://github.com/ALRhub/rkn_share)
# Copyright (c) 2021 Philipp Becker (Autonomous Learning Robots Lab @ KIT)
# licensed under MIT License
# cf. 3rd-party-licenses.txt file in the root directory of this source tree.


import torch
import numpy as np
from numpy.random import RandomState
prng = RandomState(123)
import time as t
from datetime import datetime
import os
from typing import Tuple
from torch.utils.tensorboard import SummaryWriter
from lib.utils import TimeDistributed, log_to_tensorboard, make_dir, compute_physionet_intermediate, compute_mimic_intermediate
from lib.encoder import Encoder
from lib.decoder import SplitDiagGaussianDecoder, BernoulliDecoder
from lib.CRULayer import CRULayer
from lib.CRUCell import var_activation, var_activation_inverse
from lib.losses import rmse, mse, GaussianNegLogLik, bernoulli_nll, mae
from lib.data_utils import  align_output_and_target, adjust_obs_for_extrapolation
from lib.ode_create_latent_ode_model import create_LatentODE_model
from collections import defaultdict
#from lib import grud_cell
#import lib.custom_gru as custom_gru
#import lib.peann_gru as peann_gru
#import lib.cph_gru as cph_gru
#import lib.learn_dynamics_gru as learn_dynamics_gru
#import lib.learn_discrete_dynamics_gru as learn_dynamics_gru
#from pycox.models.loss import nll_pc_hazard_loss
import pdb
from torch.distributions.normal import Normal

optim = torch.optim
nn = torch.nn
F = nn.functional


# taken from https://github.com/ALRhub/rkn_share/ and modified
class LatentODE(nn.Module):

    # taken from https://github.com/ALRhub/rkn_share/ and modified
    def __init__(self, target_dim: int, lsd: int, args, means=None, 
        use_cuda_if_available: bool = True, 
        bernoulli_output: bool = False,
        use_encoder: bool = False):
        """
        :param target_dim: output dimension
        :param lsd: latent state dimension
        :param args: parsed arguments
        :param use_cuda_if_available: if to use cuda or cpu
        :param use_bernoulli_output: if to use a convolutional decoder (for image data)
        """
        super().__init__()
        self._device = torch.device(
            "cuda" if torch.cuda.is_available() and use_cuda_if_available else "cpu")

        self._lsd = lsd
        if self._lsd % 2 == 0:
            self._lod = int(self._lsd / 2) 
        else:
            raise Exception('Latent state dimension must be even number.')
        self.args = args

        # parameters TODO: Make configurable
        self._enc_out_normalization = "pre"
        self._initial_state_variance = 10.0
        self._learning_rate = self.args.lr
        self.bernoulli_output = bernoulli_output
    
        # latent ODE specific variables
        args.latents = lsd
        args.rec_dims = self.args.lode_rec_dim#20
        args.poisson = False
        args.gen_layers = self.args.lode_gen_layers#2
        args.units = self.args.lode_units#15
        args.z0_encoder = 'odernn'
        args.rec_layers = self.args.lode_rec_layers#2
        args.gru_units = self.args.lode_gru_units#15#50
        args.classif = False
        args.linear_classif = False

        z0_prior = Normal(torch.Tensor([0.0]).to(self._device), torch.Tensor([1.]).to(self._device))
        obsrv_std = 0.01
        classif_per_tp = False
        n_labels = 1 # not sure if we need this
        self.model = create_LatentODE_model(args, target_dim, z0_prior, obsrv_std, 
            self._device, classif_per_tp=classif_per_tp, n_labels=n_labels)
        #self.model = self.model.double().to(self._device)

        '''
        self.combine_linear = nn.Linear(self._lsd, 1).to(self._device).double()
        #self.combine_linear = nn.Linear(2*self._lsd, 1).to(self._device).double()
        self.attention = grud_plus_cell.ScaledDotProductAttention(self._lsd)
        self.intermediate_linear = nn.Linear(
            self._lsd,
            #2*self._lsd,
            args.gru_intermediate_size).to(self._device).double()
        self.output_mu = nn.Linear(args.gru_intermediate_size, target_dim).to(self._device).double()
        #self.output_sd = nn.Linear(args.gru_intermediate_size, target_dim).to(self._device).double()
        self.dropout1 = nn.Dropout(p=0.3).to(self._device)
        self.dropout2 = nn.Dropout(p=0.3).to(self._device)
        self.dropout3 = nn.Dropout(p=0.3).to(self._device)
        '''

        # params and optimizer
        self._params = list(self.model.parameters())
        '''
        self._params += list(self.intermediate_linear.parameters())
        self._params += list(self.output_mu.parameters())
        '''

        self._optimizer = optim.Adam(self._params, lr=self.args.lr)
        self._shuffle_rng = np.random.RandomState(
            42)  # rng for shuffling batches

    # taken from https://github.com/ALRhub/rkn_share/ and not modified
    def _build_enc_hidden_layers(self) -> Tuple[nn.ModuleList, int]:
        """
        Builds hidden layers for encoder
        :return: nn.ModuleList of hidden Layers, size of output of last layer
        """
        raise NotImplementedError

    # taken from https://github.com/ALRhub/rkn_share/ and not modified
    def _build_dec_hidden_layers_mean(self) -> Tuple[nn.ModuleList, int]:
        """
        Builds hidden layers for mean decoder
        :return: nn.ModuleList of hidden Layers, size of output of last layer
        """
        raise NotImplementedError

    # taken from https://github.com/ALRhub/rkn_share/ and not modified
    def _build_dec_hidden_layers_var(self) -> Tuple[nn.ModuleList, int]:
        """
        Builds hidden layers for variance decoder
        :return: nn.ModuleList of hidden Layers, size of output of last layer
        """
        raise NotImplementedError

    def add_dummy_times(self, obs_batch, obs_valid, time_points, is_training, hidden_without_dummy):
        if is_training:

            sample_range_start = 1
            sample_range_end = (time_points.size(1) - 2) // 2 
            sample_range = list(range(sample_range_start, sample_range_end))
            dummy_indices = prng.choice(sample_range, size=3, replace=False)
            loss_from_dummy = 0.0
            for dummy_idx in dummy_indices:

                d_obs_batch = obs_batch.clone()
                d_obs_valid = obs_valid.clone()
                d_time_points = time_points.clone()

                # insert 5 consecutive points uniformly at random
                times_to_insert = time_points[:,dummy_idx-1][:,None] + (((time_points[:,dummy_idx] - time_points[:,dummy_idx-1]) / 6)[:,None].repeat(1,5) * torch.tensor([1,2,3,4,5]).cuda())
                d_time_points = torch.cat([time_points[:,:dummy_idx], times_to_insert, time_points[:,dummy_idx:]], 1)
                # dummy points: time_points[:,dummy_idx:dummy_idx+5]
                d_obs_batch = torch.cat([d_obs_batch[:,:dummy_idx, :], torch.zeros((obs_batch.size(0), 5, obs_batch.size(2))).double().cuda(), d_obs_batch[:,dummy_idx:, :]], 1)
                d_obs_valid = torch.cat([d_obs_valid[:,:dummy_idx, :], torch.zeros((obs_batch.size(0), 5, obs_batch.size(2))).bool().cuda(), d_obs_valid[:,dummy_idx:, :]], 1)

                hidden_w_dummy, _ = self.gru(d_obs_batch, d_obs_valid, d_time_points)
                hidden_w_dummy = torch.cat([hidden_w_dummy[:,:dummy_idx, :], hidden_w_dummy[:,dummy_idx+5:, :]], 1)
                loss_from_dummy += ((hidden_without_dummy.detach() - hidden_w_dummy) ** 2).mean()

            return loss_from_dummy
        
    
    # taken from https://github.com/ALRhub/rkn_share/ and modified
    def forward(self, obs_batch: torch.Tensor, time_points: torch.Tensor = None, 
            obs_valid: torch.Tensor = None, is_training=True, truth=None, mask_truth=None,
            numeric_event_ids=None, curr_epoch=0) -> Tuple[torch.Tensor, torch.Tensor]:
        """Single forward pass on a batch
        :param obs_batch: batch of observation sequences
        :param time_points: timestamps of observations
        :param obs_valid: boolean if timestamp contains valid observation 
        """
        # test to see if something other than GRUD logic is inc
        #obs_valid = torch.ones()
        #out, obs_hat, obs_hat_tm2 = self.gru(obs_batch, obs_valid, time_points)
        wait_until_kl_inc = 10
        if curr_epoch < wait_until_kl_inc:
            kl_coef = 0.
        else:
            kl_coef = (1-0.99**(curr_epoch - wait_until_kl_inc))

        #kl_coef = 0.1
        cut_time = 0
        observed_mask = (time_points <= cut_time) & (obs_valid.sum(axis=-1).bool())
        max_obs_ts_index = torch.where(observed_mask)[1].max().item()
        observed_data = obs_batch[:,:max_obs_ts_index+1, :]
        observed_tp = time_points[:, :max_obs_ts_index+1]
        observed_mask = obs_valid[:,:max_obs_ts_index+1,:]
        
        # to predict mask
        #mask_imput = (~obs_valid) * mask_truth
        predict_mask = (time_points > cut_time)
        min_pred_ts_index = torch.where(predict_mask)[1].min().item()
        tp_to_predict = time_points[:, min_pred_ts_index:]
        data_to_predict = truth[:,min_pred_ts_index:, :]
        to_predict_mask = mask_truth[:,min_pred_ts_index:,:]

        assert obs_batch.size(0) == 1, "make sure batch size is 1 for time dim to be flattened out"
        batch_dict = {
            'tp_to_predict' : tp_to_predict[0,:].float(),
            'observed_data': observed_data.float(),
            'observed_tp': observed_tp[0,:].float(),
            'observed_mask': observed_mask,
            'data_to_predict': data_to_predict, 
            'mask_predicted_data': to_predict_mask,
            'labels': None,
            'mode': 'extrap' 
        }
        result = self.model.compute_all_losses(batch_dict, n_traj_samples = 3, kl_coef = kl_coef, 
                    numeric_event_ids=numeric_event_ids)

        # output an image
        if self.bernoulli_output:
            out_mean = self._dec(post_mean)
            out_var = None

        # output prediction for the next time step
        elif self.args.task == 'one_step_ahead_prediction':
            out_mean, out_var = self._dec(
                prior_mean, torch.cat(prior_cov, dim=-1))

        # output filtered observation
        else:
            # need to figure this out
            out_mean = result['output_mean']
            out_var = result['output_var']

        intermediates = {
            'loss': result['loss']
            #'obs_hat': obs_hat,
            #'combo_weight': comb_weight,
            #'loss_from_dummy_inserts': loss_from_dummy_inserts
            #'obs_hat_tm2': obs_hat_tm2
        }

        return out_mean, out_var, intermediates, data_to_predict, to_predict_mask

    # new code component
    def interpolation(self, data, track_gradient=True):
        """Computes loss on interpolation task

        :param data: batch of data
        :param track_gradient: if to track gradient for backpropagation
        :return: loss, outputs, inputs, intermediate variables, metrics on imputed points
        """
        if self.bernoulli_output:
            obs, truth, obs_valid, obs_times, mask_truth = [
                j.to(self._device) for j in data]
            mask_obs = None
        else:
            obs, truth, obs_valid, obs_times, mask_truth, mask_obs = [
                j.to(self._device) for j in data]

        obs_times = self.args.ts * obs_times

        with torch.set_grad_enabled(track_gradient):
            output_mean, output_var, intermediates = self.forward(
                obs_batch=obs, time_points=obs_times, obs_valid=obs_valid)

            if self.bernoulli_output:
                loss = bernoulli_nll(truth, output_mean, uint8_targets=False)
                mask_imput = (~obs_valid[...,None, None, None]) * mask_truth
                imput_loss = np.nan #TODO: compute bernoulli loss on imputed points
                imput_mse = mse(truth.flatten(start_dim=2), output_mean.flatten(start_dim=2), mask=mask_imput.flatten(start_dim=2))

            else:
                loss = GaussianNegLogLik(
                    output_mean, truth, output_var, mask=mask_truth)
                '''
                surv_loss = nll_pc_hazard_loss(intermediates['surv_outputs']['surv_phi'],
                            intermediates['surv_outputs']['surv_idx_durations'],
                            intermediates['surv_outputs']['surv_events'],
                            intermediates['surv_outputs']['surv_interval_frac'])
                intermediates['surv_loss'] = surv_loss.item()
                intermediates['output_loss'] = loss.item()
                loss = loss + self.args.peann_gamma * surv_loss
                '''
                # compute metric on imputed points only
                mask_imput = (~obs_valid[...,None]) * mask_truth
                imput_loss = GaussianNegLogLik(output_mean, truth, output_var, mask=mask_imput)
                imput_mse = mse(truth, output_mean, mask=mask_imput)
        
        return loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates, imput_loss, imput_mse

    # new code component
    def extrapolation(self, data, track_gradient=True, curr_epoch=0):
        """Computes loss on extrapolation task

        :param data: batch of data
        :param track_gradient: if to track gradient for backpropagation
        :return: loss, outputs, inputs, intermediate variables, metrics on imputed points
        """
        obs, truth, obs_valid, obs_times, mask_truth, mask_obs, numeric_event_ids = [
            j.to(self._device) for j in data]

        if numeric_event_ids.ndim > 1:
            assert numeric_event_ids.ndim==2, "more than two dimensions in numeric event ids"
            numeric_event_ids = numeric_event_ids[0,:]

        obs, obs_valid_extrap, obs_valid = adjust_obs_for_extrapolation(self.args.dataset, obs, obs_valid, mask_obs, obs_times)
        '''
        mask_obs = mask_truth.clone()
        extrapolation_mask = (obs_times[:,:,None].repeat(1,1,obs.shape[-1]) <= 0.0) 
        obs_valid = mask_truth.bool().clone() * extrapolation_mask
        obs = torch.where(obs_valid, obs, 0.)
        assert obs_times[(obs != truth).any(axis=-1)].min().item() > 0.0, \
            "time cheating"
        '''
        '''
        obs, obs_valid = adjust_obs_for_extrapolation(
            obs, obs_valid, obs_times, self.args.cut_time)
        obs_times = self.args.ts * obs_times
        '''

        obs_times = obs_times * 0.00000001
        with torch.set_grad_enabled(track_gradient):
            # output_mean and output var are already masked to impute ids
            output_mean, output_var, intermediates, data_to_predict, to_predict_mask = self.forward(
                obs_batch=obs, time_points=obs_times, obs_valid=obs_valid, 
                is_training=track_gradient, truth=truth, mask_truth=mask_truth, 
                numeric_event_ids=numeric_event_ids, curr_epoch=curr_epoch)

            loss = intermediates['loss']
            '''
            loss = mse(truth[:,:,numeric_event_ids], output_mean[:,:,numeric_event_ids], 
                    mask=mask_truth[:,:,numeric_event_ids])
            '''

            '''
            if track_gradient:
                loss = loss + loss_from_dummy_inserts
            '''
            
            # compute metric on imputed points only
            mask_imput = (~obs_valid) * mask_truth
            imput_loss = GaussianNegLogLik(
                data_to_predict[:,:,numeric_event_ids], output_mean[:,:,numeric_event_ids], 
                output_var[:,:,numeric_event_ids], mask=to_predict_mask[:,:,numeric_event_ids])
            imput_mse = mse(data_to_predict[:,:,numeric_event_ids], 
                output_mean[:,:,numeric_event_ids], 
                mask=to_predict_mask[:,:,numeric_event_ids])
            imput_mae = mae(data_to_predict[:,:,numeric_event_ids], output_mean[:,:,numeric_event_ids], 
                mask=to_predict_mask[:,:,numeric_event_ids])

        intermediates = {}
        next_step_impute_loss = None

        # construct output_mean for full sequence for downsteam processing:
        # MSE is bogus
        # Impute stats are good
        output_mean2 = torch.zeros_like(obs).float()
        output_mean2[:,obs_times[0,:] >0,:] = output_mean
        return loss, output_mean2, output_var, obs, truth, mask_obs, mask_truth, intermediates, imput_loss, imput_mse, mask_imput, numeric_event_ids, imput_mae, next_step_impute_loss

    # new code component
    def regression(self, data, track_gradient=True):
        """Computes loss on regression task

        :param data: batch of data
        :param track_gradient: if to track gradient for backpropagation
        :return: loss, input, intermediate variables and computed output
        """
        obs, truth, obs_times, obs_valid = [j.to(self._device) for j in data]
        mask_truth = None
        mask_obs = None
        with torch.set_grad_enabled(track_gradient):
            output_mean, output_var, intermediates = self.forward(
                obs_batch=obs, time_points=obs_times, obs_valid=obs_valid)
            loss = GaussianNegLogLik(
                output_mean, truth, output_var, mask=mask_truth)

        return loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates

    # new code component
    def one_step_ahead_prediction(self, data, track_gradient=True):
        """Computes loss on one-step-ahead prediction

        :param data: batch of data
        :param track_gradient: if to track gradient for backpropagation
        :return: loss, input, intermediate variables and computed output
        """
        obs, truth, obs_valid, obs_times, mask_truth, mask_obs = [
            j.to(self._device) for j in data]
        obs_times = self.args.ts * obs_times
        with torch.set_grad_enabled(track_gradient):
            output_mean, output_var, intermediates = self.forward(
                obs_batch=obs, time_points=obs_times, obs_valid=obs_valid)
            output_mean, output_var, truth, mask_truth = align_output_and_target(
                output_mean, output_var, truth, mask_truth)
            loss = GaussianNegLogLik(
                output_mean, truth, output_var, mask=mask_truth)

        return loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates

    # new code component
    def train_epoch(self, dl, optimizer, curr_epoch=0):
        """Trains model for one epoch 

        :param dl: dataloader containing training data
        :param optimizer: optimizer to use for training
        :return: evaluation metrics, computed output, input, intermediate variables
        """
        epoch_ll = 0
        epoch_rmse = 0
        epoch_mse = 0

        if self.args.save_intermediates is not None:
            mask_obs_epoch = []
            intermediates_epoch = []

        if self.args.task == 'extrapolation' or self.args.task == 'interpolation':
            epoch_imput_ll = 0
            epoch_imput_mse = 0

        for i, data in enumerate(dl):

            #print('processing: {} / {}'.format(i, len(dl)))
            if self.args.task == 'interpolation':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates, imput_loss, imput_mse = self.interpolation(
                    data)

            elif self.args.task == 'extrapolation':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates, imput_loss, imput_mse, \
                    mask_imput, numeric_event_ids, _, next_step_impute_loss = self.extrapolation(data, curr_epoch=curr_epoch)

            elif self.args.task == 'regression':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates = self.regression(
                    data)

            elif self.args.task == 'one_step_ahead_prediction':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates = self.one_step_ahead_prediction(
                    data)

            else:
                raise Exception('Unknown task')

            loss = loss

            # check for NaNs
            if torch.any(torch.isnan(loss)):
                print('--NAN in loss')
            for name, par in self.named_parameters():
                if torch.any(torch.isnan(par)):
                    print('--NAN before optimiser step in parameter ', name)
            torch.autograd.set_detect_anomaly(
                self.args.anomaly_detection)

            # backpropagation
            optimizer.zero_grad()
            loss.backward()
            if self.args.grad_clip:
                nn.utils.clip_grad_norm_(self.parameters(), 1)
            optimizer.step()

            # check for NaNs in gradient
            for name, par in self.named_parameters():
                if name in ['gru.event_ids', 'gru.hidden_ids']:
                    # stored event ids are static and will not have gradients
                    continue
                #print(name)
                if torch.any(torch.isnan(par.grad)):
                    print('--NAN in gradient ', name)
                if torch.any(torch.isnan(par)):
                    print('--NAN after optimiser step in parameter ', name)

            # aggregate metrics and intermediates over entire epoch
            epoch_ll += loss.item()
            epoch_rmse += rmse(truth[...,numeric_event_ids], output_mean[...,numeric_event_ids], 
                mask_truth[...,numeric_event_ids]).item()
            epoch_mse += mse(truth[...,numeric_event_ids], output_mean[...,numeric_event_ids], 
                mask_truth[...,numeric_event_ids]).item()

            if self.args.task == 'extrapolation' or self.args.task == 'interpolation':
                epoch_imput_ll += imput_loss.item()
                epoch_imput_mse += imput_mse.item()
                imput_metrics = [epoch_imput_ll/(i+1), epoch_imput_mse/(i+1)]
            else:
                imput_metrics = None

            if self.args.save_intermediates is not None:
                intermediates_epoch.append(intermediates)
                mask_obs_epoch.append(mask_obs)

        # save for plotting
        if self.args.save_intermediates is not None:
            torch.save(mask_obs_epoch, os.path.join(
                self.args.save_intermediates, 'train_mask_obs.pt'))
            torch.save(intermediates_epoch, os.path.join(
                self.args.save_intermediates, 'train_intermediates.pt'))

        return epoch_ll/(i+1), epoch_rmse/(i+1), epoch_mse/(i+1), [output_mean, output_var], intermediates, [obs, truth, mask_obs], imput_metrics

    # new code component
    def eval_epoch(self, dl, wandb=None, curr_epoch=0):
        """Evaluates model on the entire dataset

        :param dl: dataloader containing validation or test data
        :return: evaluation metrics, computed output, input, intermediate variables
        """
        epoch_ll = 0
        epoch_rmse = 0
        epoch_mse = 0
        dynamics_mse = 0
        intermediate_results = {}

        if self.args.task == 'extrapolation' or self.args.task == 'interpolation':
            epoch_imput_ll = 0
            epoch_imput_mse = 0
            epoch_imput_mae = 0

        if self.args.save_intermediates is not None:
            mask_obs_epoch = []
            intermediates_epoch = []

        for i, data in enumerate(dl):

            if self.args.task == 'interpolation':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates, imput_loss, imput_mse = self.interpolation(
                    data, track_gradient=False)

            elif self.args.task == 'extrapolation':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates, imput_loss, imput_mse, mask_imput, numeric_event_ids, imput_mae, next_step_impute_loss = self.extrapolation(
                    data, track_gradient=False, curr_epoch=curr_epoch)

            elif self.args.task == 'regression':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates = self.regression(
                    data, track_gradient=False)

            elif self.args.task == 'one_step_ahead_prediction':
                loss, output_mean, output_var, obs, truth, mask_obs, mask_truth, intermediates = self.one_step_ahead_prediction(
                    data, track_gradient=False)

            epoch_ll += loss
            epoch_rmse += rmse(truth[..., numeric_event_ids], output_mean[..., numeric_event_ids], 
                mask_truth[..., numeric_event_ids]).item()
            epoch_mse += mse(truth[..., numeric_event_ids], output_mean[..., numeric_event_ids], mask_truth[..., numeric_event_ids]).item()
            '''
            decayed_h_states = intermediates['surv_outputs']['decayed_hidden_states']
            h_states = intermediates['surv_outputs']['hidden_states']
            dynamics_mse += mse(h_states[:,1:,:].detach(), 
                decayed_h_states[:, :-1, :]).item()
            '''

            if self.args.task == 'extrapolation' or self.args.task == 'interpolation':
                if self.args.dataset == 'physionet':
                    intermediate_results = compute_physionet_intermediate(mse, mask_imput,
                        truth,output_mean, intermediate_results)
                elif self.args.dataset == 'mimic':
                    intermediate_results = compute_mimic_intermediate(mse, mask_imput,
                        truth,output_mean, intermediate_results)

                # weights that combine the attention and GRU estimates
                #intermediate_results['combo_weights_all'].append(intermediates['combo_weight'])
                epoch_imput_ll += imput_loss
                epoch_imput_mse += imput_mse
                epoch_imput_mae += imput_mae
                imput_metrics = [epoch_imput_ll/(i+1), epoch_imput_mse/(i+1), epoch_imput_mae/(i+1)]

            else:
                imput_metrics = None

            if self.args.save_intermediates is not None:
                intermediates_epoch.append(intermediates)
                mask_obs_epoch.append(mask_obs)

        # normalize by batch size
        for k, v in intermediate_results.items():
            if k != 'combo_weights_all':
                intermediates[k] = v / (i+1)
            else:
                comb_weights = intermediate_results['combo_weights_all']
                max_ts = np.max([comb_weights[i].shape[1] for i in range(len(comb_weights))])
                stats_by_ts_index = defaultdict(list)
                mean_comb_by_ts_index = {}
                for ts_index in range(max_ts):
                    for batch_comb_weights in comb_weights:
                        if ts_index < batch_comb_weights.shape[1]:
                            stats_by_ts_index[ts_index].append(batch_comb_weights[:,ts_index])
                for ts_index in range(max_ts):
                    mean_comb_by_ts_index[ts_index] = np.concatenate(stats_by_ts_index[ts_index]).mean()
                data = [[ts_index, mean] for (ts_index, mean) in mean_comb_by_ts_index.items()]
                wandb_table = wandb.Table(data=data, columns=['ts_index', 'combo_weight_gru'])
                wandb_plot = wandb.plot.line(wandb_table, "ts_index", "combo_weight_gru", title='GRU combo weight over time indexes')
                intermediates['combo_wts_gru_by_ts_index'] = wandb_table
                intermediates['combo_wts_gru_by_ts_index_plot'] = wandb_plot
                    
                
        # save for plotting
        if self.args.save_intermediates is not None:
            torch.save(output_mean, os.path.join(
                self.args.save_intermediates, 'valid_output_mean.pt'))
            torch.save(obs, os.path.join(
                self.args.save_intermediates, 'valid_obs.pt'))
            torch.save(output_var, os.path.join(
                self.args.save_intermediates, 'valid_output_var.pt'))
            torch.save(truth, os.path.join(
                self.args.save_intermediates, 'valid_truth.pt'))
            torch.save(intermediates_epoch, os.path.join(
                self.args.save_intermediates, 'valid_intermediates.pt'))
            torch.save(mask_obs_epoch, os.path.join(
                self.args.save_intermediates, 'valid_mask_obs.pt'))

        return epoch_ll/(i+1), epoch_rmse/(i+1), epoch_mse/(i+1), [output_mean, output_var], intermediates, [obs, truth, mask_obs], imput_metrics

    # new code component
    def run_train(self, train_dl, valid_dl, identifier, logger, epoch_start=0, wandb=None):
        """Trains model on trainset and evaluates on test data. Logs results and saves trained model.

        :param train_dl: training dataloader
        :param valid_dl: validation dataloader
        :param identifier: logger id
        :param logger: logger object
        :param epoch_start: starting epoch
        """

        optimizer = optim.Adam(self.parameters(), self.args.lr)
        def lr_update(epoch): return self.args.lr_decay ** epoch
        scheduler = torch.optim.lr_scheduler.LambdaLR(
            optimizer, lr_lambda=lr_update)
        
        make_dir(f'results/tensorboard/{self.args.dataset}')
        writer = SummaryWriter(f'results/tensorboard/{self.args.dataset}/{identifier}')

        for epoch in range(epoch_start, self.args.epochs):
            start = datetime.now()
            logger.info(f'Epoch {epoch} starts: {start.strftime("%H:%M:%S")}')

            # train
            train_ll, train_rmse, train_mse, train_output, intermediates, train_input, train_imput_metrics = self.train_epoch(
                train_dl, optimizer, curr_epoch=epoch)
            end_training = datetime.now()
            if self.args.tensorboard:
                log_to_tensorboard(self, writer=writer,
                                mode='train',
                                metrics=[train_ll, train_rmse, train_mse],
                                output=train_output,
                                input=train_input,
                                intermediates=intermediates,
                                epoch=epoch,
                                imput_metrics=train_imput_metrics,
                                log_rythm=self.args.log_rythm)

            # eval
            valid_ll, valid_rmse, valid_mse, valid_output, valid_intermediates, valid_input, valid_imput_metrics = self.eval_epoch(
                valid_dl, wandb=wandb, curr_epoch=epoch)
            if self.args.tensorboard:
                log_to_tensorboard(self, writer=writer,
                                mode='valid',
                                metrics=[valid_ll, valid_rmse, valid_mse],
                                output=valid_output,
                                input=valid_input,
                                intermediates=intermediates,
                                epoch=epoch,
                                imput_metrics=valid_imput_metrics,
                                log_rythm=self.args.log_rythm)

            end = datetime.now()
            logger.info(f'Training epoch {epoch} took: {(end_training - start).total_seconds()}')
            logger.info(f'Epoch {epoch} took: {(end - start).total_seconds()}')
            logger.info(f' train_nll: {train_ll:3f}, train_mse: {train_mse:3f}')
            logger.info(f' valid_nll: {valid_ll:3f}, valid_mse: {valid_mse:3f}')
            wandb_dict = {}
            wandb_dict['train_nll'] = train_ll
            wandb_dict['valid_nll'] = valid_ll
            wandb_dict['train_mse'] = train_mse
            wandb_dict['valid_mse'] = valid_mse
            if self.args.task == 'extrapolation' or self.args.impute_rate is not None:
                if self.bernoulli_output:
                    logger.info(f' train_mse_imput: {train_imput_metrics[1]:3f}')
                    logger.info(f' valid_mse_imput: {valid_imput_metrics[1]:3f}')
                else:
                    logger.info(f' train_nll_imput: {train_imput_metrics[0]:3f}, train_mse_imput: {train_imput_metrics[1]:3f}')
                    logger.info(f' valid_nll_imput: {valid_imput_metrics[0]:3f}, valid_mse_imput: {valid_imput_metrics[1]:3f}')
                    wandb_dict['train_nll_imput'] = train_imput_metrics[0]
                    wandb_dict['valid_nll_imput'] = valid_imput_metrics[0]
                    wandb_dict['train_mse_imput'] = train_imput_metrics[1]
                    wandb_dict['valid_mse_imput'] = valid_imput_metrics[1]
                    wandb_dict['valid_mae_imput'] = valid_imput_metrics[2]
                    for k, v in valid_intermediates.items():
                        wandb_dict[k] = v
            if self.args.log_wandb:
                wandb.log(wandb_dict)

            scheduler.step()
        
        make_dir(f'results/models/{self.args.dataset}')
        '''
        torch.save({'epoch': epoch,
                    'model_state_dict': self.state_dict(),
                    'optimizer_state_dict': optimizer.state_dict(),
                    'loss': train_ll,
                    }, f'../results/models/{self.args.dataset}/{identifier}.tar')
        '''
        torch.save(self, f'results/models/{self.args.dataset}/lode_{identifier}.tar')
