
import os
import sys
import json
import math
import copy
import random
from time import perf_counter
import numpy as np

import torch
import torchvision
import torch.nn as nn
from torch.autograd import Variable
from torch.nn.utils import clip_grad_norm_
from torchvision.models import *

codebase = os.path.abspath(os.path.dirname(os.path.dirname(__file__)))
sys.path.append(codebase)
from trainers import BaseTrainer, read_options

import warnings
warnings.filterwarnings("ignore")
    
class Trainer(BaseTrainer):
    def __init__(self, params):
        super(Trainer, self).__init__(params)
        
        for key in ['test_acc', 'train_acc', 'test_loss']:
            setattr(self, key, [])
        self.epoch_start = 0
        
        if self.continue_from is not None:
            self.log_name = os.path.join(os.path.dirname(__file__), '../logs', self.continue_from)
            results = json.load(open(self.log_name+"/results.json", 'r'))
            
            for key in ['test_acc', 'train_acc', 'test_loss']:
                setattr(self, key, results[key])
            
            for file in os.listdir(self.log_name):
                if file.endswith(".pt"):
                    self.epoch_start = int(file.split('.pt')[0][5:])
                    self.model.load_state_dict(torch.load(self.log_name+"/"+file))
            print('continue from', self.log_name, 'epoch', self.epoch_start)
        
        
    def train(self):
        total_step = len(self.train_loader)
        test_interval = total_step // self.num_test_per_epoch
        
        tmp_g = torch.cat([p.data.clone().view(-1) for _, p in self.model.named_parameters()])
        self.D = len(tmp_g)
        del tmp_g

        avg_iter_time = 0
        noise = dict()
        
        for epoch in range(self.epoch_start, self.epochs):
            
            if epoch % self.eval_every_epoch == 0:
                L_test = self.get_test_loss()
                train_accu = self.get_train_accuracy()
                print('epoch {} test loss {:.5f} train accuracy {:.5f}'.format(epoch, L_test, train_accu), flush=True)
                self.train_acc.append(train_accu)
                self.test_loss.append(L_test)
                self.model.train()

            for i, (xs, ys) in enumerate(self.train_loader):
                t1 = perf_counter()
                
                if i % test_interval == 0 and i > 0:
                    test_accu = self.get_test_accuracy()
                    print('epoch', epoch, 'iter', i, 'test accuracy', test_accu)
                    self.test_acc.append(test_accu)
                    self.model.train()

                xs = xs.to(self.device)
                ys = ys.to(self.device)
                B = xs.size(0)
                
                overall_grad = dict()
                for p_name, p in self.model.named_parameters():
                    overall_grad[p_name] = torch.zeros_like(p)
                
                self.model.zero_grad()
                
                for _ in range(self.num_directions): # sample directions
                    for p_name, p in self.model.named_parameters():
                        noise[p_name] = torch.randn_like(p.data)
                        p.data = p.data + noise[p_name] * self.perturbation_scale 

                    with torch.no_grad():
                        predicted1 = self.model(xs)
                        l1 = self.loss_flat(predicted1, ys)
                    
                    for p_name, p in self.model.named_parameters():
                        p.data = p.data - 2 * noise[p_name] * self.perturbation_scale       
                    
                    with torch.no_grad():
                        predicted2 = self.model(xs)
                        l2 = self.loss_flat(predicted2, ys)

                    l = 0.5 * (l1 - l2) / self.perturbation_scale
                    l = torch.mean(l)
                    
                    for p_name, p in self.model.named_parameters():
                        overall_grad[p_name].add_(l * noise[p_name])
                        p.data = p.data + noise[p_name] * self.perturbation_scale  # restore to original
                
                for p_name, p in self.model.named_parameters():
                    overall_grad[p_name] = overall_grad[p_name] / self.num_directions
                
                for p_name, p in self.model.named_parameters():
                    p.grad = overall_grad[p_name] 
                
                
                self.optimizer.step()  # apply p.grad
                
                if epoch==0 and i<20:
                    avg_iter_time += perf_counter() - t1
                    if i==19:
                        avg_iter_time /= 20
                        print('avg iter time', avg_iter_time, 's')
                        # return
            
            json.dump({key: eval(f'self.{key}') for key in ['test_acc', 'train_acc', 'test_loss']}, 
                        open(self.log_name+"/results.json", 'w'), indent=4)
            torch.save(self.model.state_dict(), self.log_name+"/epoch"+str(epoch+1)+".pt")
            for file in [self.log_name+"/epoch"+str(epoch)+".pt"]:
                if os.path.exists(file):
                    os.remove(file)
    
    
def main():
    options = read_options()
    t = Trainer(options)
    t.train()


if __name__ == "__main__":
    main()
    
    
