import numpy as np
from gym import utils
from gym.envs.mujoco import mujoco_env
from gym.envs.registration import EnvSpec


class AntMOPOMassDampEnv(mujoco_env.MujocoEnv, utils.EzPickle):
    # def __init__(self, goal=15.0/180*np.pi):
    def __init__(self, goal=30.0/180*np.pi, mass_scale_set=[0.75, 1.0, 1.25], damping_scale_set=[0.75, 1.0, 1.25]):
        self._goal = goal
        mujoco_env.MujocoEnv.__init__(self, 'ant.xml', 5)
        self.spec = EnvSpec('Ant-v2')

        self.original_mass = np.copy(self.model.body_mass)
        self.original_damping = np.copy(self.model.dof_damping)

        self.mass_scale_set = mass_scale_set
        self.damping_scale_set = damping_scale_set
        utils.EzPickle.__init__(self, goal, mass_scale_set, damping_scale_set)

    def _set_observation_space(self, observation):
        super(AntMOPOMassDampEnv, self)._set_observation_space(observation)
        proc_observation = self.obs_preproc(observation[None])
        self.proc_observation_space_dims = proc_observation.shape[-1]

    def step(self, a):
        # self.render()
        xy_position_before = self.get_body_com("torso")[:2].copy()
        self.do_simulation(a, self.frame_skip)
        xy_position_after = self.get_body_com("torso")[:2].copy()
        direct = (np.cos(self._goal), np.sin(self._goal))

        xy_velocity = (xy_position_after - xy_position_before) / self.dt
        x_velocity, y_velocity = xy_velocity

        # xposbefore = self.get_body_com("torso")[0]
        # self.do_simulation(a, self.frame_skip)
        # xposafter = self.get_body_com("torso")[0]
        
        # forward_reward = (xposafter - xposbefore)/self.dt
        forward_reward = x_velocity
        angle_reward = np.dot(np.array(xy_velocity), direct)
        ctrl_cost = .5 * np.square(a).sum()
        contact_cost = 0.5 * 1e-3 * np.sum(
            np.square(np.clip(self.sim.data.cfrc_ext, -1, 1)))
        survive_reward = 1.0
        reward = forward_reward - ctrl_cost - contact_cost + survive_reward
        state = self.state_vector()
        notdone = np.isfinite(state).all() \
            and state[2] >= 0.2 and state[2] <= 1.0
        done = not notdone
        ob = self._get_obs()
        return ob, reward, done, dict(
            reward_ctrl=-ctrl_cost,
            reward_contact=-contact_cost,
            reward_survive=survive_reward,
            reward_forward=forward_reward,
            reward_angle=angle_reward,
            x_position=xy_position_after[0],
            y_position=xy_position_after[1])

    def seed(self, seed=None):
        if seed is None:
            self._seed = 0
        else:
            self._seed = seed
        super().seed(seed)

    def _get_obs(self):
        return np.concatenate([
            self.sim.data.qpos.flat[2:],
            self.sim.data.qvel.flat,
            # np.clip(self.sim.data.cfrc_ext, -1, 1).flat,
        ])

    def obs_preproc(self, obs):
        if isinstance(obs, np.ndarray):

            return obs
        else:
            raise NotImplementedError

    def obs_postproc(self, obs, pred):
        if isinstance(obs, np.ndarray):
            return obs + pred
        else:
            raise NotImplementedError

    def reset_model(self):
        qpos = self.init_qpos + self.np_random.uniform(size=self.model.nq, low=-.001, high=.001)
        qvel = self.init_qvel + self.np_random.randn(self.model.nv) * .001
        self.set_state(qpos, qvel)

        random_index = self.np_random.randint(len(self.mass_scale_set))
        self.mass_scale = self.mass_scale_set[random_index]

        random_index = self.np_random.randint(len(self.damping_scale_set))
        self.damping_scale = self.damping_scale_set[random_index]

        self.change_env()
        return self._get_obs()

    def change_env(self):
        mass = np.copy(self.original_mass)
        damping = np.copy(self.original_damping)
        mass *= self.mass_scale
        damping *= self.damping_scale

        self.model.body_mass[:] = mass
        self.model.dof_damping[:] = damping

    def change_mass(self, mass):
        self.mass_scale = mass

    def change_damping(self, damping):
        self.damping_scale = damping

    def viewer_setup(self):
        self.viewer.cam.distance = self.model.stat.extent * 0.5
