import pandas as pd
import attrs

from sandbagging.pb_data_types import PbsSplits


def make_splits_for_fracs(df, fracs):
    assert sum(fracs) == 1.0
    splits = []
    start = 0
    for i, frac in enumerate(fracs):
        end = start + int(len(df) * frac)
        if i == len(fracs) - 1:
            assert abs(end - len(df)) < 5, "shouldn't be very off..."
            end = len(df)
        splits.append(df.iloc[start:end].reset_index(drop=True))
        start = end

    assert start == len(df)

    return splits


@attrs.frozen
class SplitGetterConfig:
    train_baseline_frac: float = 0.1
    train_lock_frac: float = 0.4
    train_elicit_frac: float = 0.3
    val_frac: float = 0.2

    def __attrs_post_init__(self):
        assert self.train_baseline_frac > 0
        assert self.train_lock_frac > 0
        assert self.train_elicit_frac > 0
        assert self.val_frac > 0
        assert self.val_frac + self.train_baseline_frac + self.train_lock_frac + self.train_elicit_frac == 1.0


def make_split_getter_for_test_exclusive_splitting(
    train: pd.DataFrame, test: pd.DataFrame, config: SplitGetterConfig = SplitGetterConfig()
):
    def get_splits(seed: int):
        count_val = int((len(train) + len(test)) * config.val_frac)

        assert count_val <= len(test)

        shuf_test = test.sample(frac=1.0, random_state=seed).reset_index(drop=True)

        val_split = shuf_test.iloc[:count_val].reset_index(drop=True)
        test_for_train = shuf_test.iloc[count_val:].reset_index(drop=True)

        # 3 is arbitrary
        train_split = pd.concat([train, test_for_train]).sample(frac=1.0, random_state=seed + 3).reset_index(drop=True)

        total_train_frac = 1.0 - config.val_frac
        norm_train_baseline_frac = config.train_baseline_frac / total_train_frac
        norm_train_lock_frac = config.train_lock_frac / total_train_frac
        norm_train_elicit_frac = config.train_elicit_frac / total_train_frac

        train_baseline_split, train_lock_split, train_elicit_split = make_splits_for_fracs(
            train_split,
            [norm_train_baseline_frac, norm_train_lock_frac, norm_train_elicit_frac],
        )

        return PbsSplits.from_dfs(
            val_split=val_split,
            train_baseline_split=train_baseline_split,
            train_lock_split=train_lock_split,
            train_elicit_split=train_elicit_split,
        )

    return get_splits
