import asyncio
import logging
import os
import shutil
from typing import Any, Coroutine, Optional, TypeVar

import attrs
from tqdm import tqdm

from sandbagging.utils.loaded_problem import LoadedProblem
from sandbagging.utils.json_cache import json_converter
from sandbagging.utils.redis_cache_wrapper import get_event_loop


@attrs.frozen
class ProblemIdx:
    idx: int

    def g(self):
        return global_constant_probs[self.idx]


cache_dir = "/".join(__file__.split("/")[:-1]) + "/sandbagging_cache_c"

if not os.path.exists(cache_dir) or not os.path.exists(f"{cache_dir}/problems.jsonl"):
    os.makedirs(cache_dir, exist_ok=True)
    logging.warning(f"No problems found at {cache_dir}")
    global_constant_probs = []
else:
    with open(f"{cache_dir}/problems.jsonl", "r") as f:
        global_constant_probs = [json_converter.loads(s, LoadedProblem) for s in f.readlines()]

T = TypeVar("T")


async def gather_tqdm(coroutines: list[Coroutine[Any, Any, T]], desc="", batch_size: Optional[int] = None) -> list[T]:
    pbar = tqdm(total=len(coroutines), desc=desc)

    async def wrap(coroutine):
        r = await coroutine
        with pbar.get_lock():
            pbar.update(1)
        return r

    batches = (
        [coroutines]
        if batch_size is None
        else [coroutines[i : i + batch_size] for i in range(0, len(coroutines), batch_size)]
    )

    all_res = []
    for batch in batches:
        res = await asyncio.gather(*[wrap(coroutine=c) for c in batch])
        all_res += res
    pbar.close()
    return all_res


def test_gather_tqdm():
    def get_coroutines():
        cs = []
        for i in range(10):

            async def c(i=i):
                await asyncio.sleep(0.1 * (5 - i) ** 2)
                return i

            cs.append(c())
        return cs

    async def run():
        assert await gather_tqdm(get_coroutines()) == await asyncio.gather(*get_coroutines())

    asyncio.run(run())

    def get_coroutines():
        cs = []
        for i in range(100):

            async def c(i=i):
                await asyncio.sleep(0.001 * (5 - i) ** 2)
                return i

            cs.append(c())
        return cs

    async def run():
        assert await gather_tqdm(get_coroutines(), batch_size=32) == await asyncio.gather(*get_coroutines())

    asyncio.run(run())


def asyncio_run(coro: Coroutine[Any, Any, T]) -> T:
    loop = get_event_loop()
    return loop.run_until_complete(coro)


def tmp_dir_of_gpu(fast_temp_dir: str) -> str:
    gpu = int(os.environ["CUDA_VISIBLE_DEVICES"])  # only works for processes with one GPU
    d = f"{fast_temp_dir}/{gpu}"
    assert d.startswith("/tmp"), "Safeguard, remove at your own risk"
    shutil.rmtree(d, ignore_errors=True)
    os.makedirs(d, exist_ok=True)
    return d


if __name__ == "__main__":

    async def c():
        await asyncio.sleep(0.1)
        return 5

    print(asyncio_run(c()))
