import asyncio
from collections import Counter

from datasets import load_dataset

from sandbagging.utils.loaded_problem import InputOutputTestCases, LoadedProblem
from sandbagging.utils.info_types import ProblemSolutionOut, TestCaseOutcome
from sandbagging.utils.json_cache import json_converter
from sandbagging.utils.models_for_monitoring import ChatInput, smart_models
from sandbagging.code_data import extract_solution, get_test_case_outcomes
from sandbagging.dataset_wrapping import cache_dir, gather_tqdm

mbpp = load_dataset("mbpp", "full")
all_examples = list(mbpp["train"]) + list(mbpp["validation"]) + list(mbpp["test"])


def example_to_chat_input(ex):
    tests = "\n".join(ex["test_list"])
    prompt = f"""
Convert the following python problem statement and test into a language agnostic problem description and input-output test as well as a c++ solution.

Problem statement: {ex["text"]}
Python solution:
```python
{ex["code"]}
```
Python test:
```python
{tests}
```

Your output should look like this:

-----Problem-----
<problem description. In a format like "You are given n distinct integers. Find the number of pairs of integers that sum to a given value k." do NOT say "Write a function...".>
-----Input-----
<description of the expect input of the program. In the format "The first line of the input contains a single integer n. The second line contains...">
-----Output-----
<description of the expected output of the program. In the format "The first line of the output should contains a single integer, the number of ...">
-----Examples-----
Input
<first line>
<second line>
...
Output
<first line>
...
Input
<first line>
...
Output
<first line>
...
-----Solution-----
```cpp
<the solution, should contain a main function that reads the input and prints the output>
```

Feel free to adapt the inputs and outputs to formats that are more natural in the context of C++ cin and cout.
Say "This problem can't be converted" if you think the problem can't be converted.
"""
    return ChatInput([("user", prompt)])


# dumb_conv = await smart_models(example_to_chat_input(mbpp["train"][0]))
# print(dumb_conv[0][0].completion)

START_ID = 1_000_000
MAX_N_PB = 10_000


async def convert_all():
    all_responses = await gather_tqdm(
        [smart_models(example_to_chat_input(ex)) for ex in all_examples[:MAX_N_PB]], desc="gen_solutions"
    )

    problems: list[LoadedProblem] = []
    for i, (response, _) in enumerate(all_responses):
        completion = response[0].completion

        # parse completion
        try:
            description = completion.split("-----Problem-----")[1].split("-----Examples-----")[0].strip()

            tests = completion.split("-----Examples-----")[1].split("-----Solution-----")[0].strip()
            input_and_outputs = tests.split("Input")[1:]
            inputs = [inp.split("Output")[0].strip() + "\n" for inp in input_and_outputs]
            outputs = [out.split("Output")[1].strip() + "\n" for out in input_and_outputs]
            solution = extract_solution(completion.split("-----Solution-----")[1].strip())

            problems.append(
                LoadedProblem(
                    task_id=START_ID + i,
                    dataset="mbpp",
                    description=description,
                    solutions=[solution],
                    base_code="",
                    base_code_fmt_example="",
                    test_cases=InputOutputTestCases(inputs=inputs, outputs=outputs),
                    difficulty="mbpp",
                )
            )
        except Exception as e:
            print(e)
            continue

    correctness = await gather_tqdm(
        [
            get_test_case_outcomes(
                ProblemSolutionOut(problem.description, problem.solutions[0]),
                problem.test_cases,
            )
            for problem in problems
        ],
        desc="get_test_case_outcomes",
    )
    # print compilation failures
    for problem, (correctness_result, err) in zip(problems, correctness):
        if correctness_result[0] == TestCaseOutcome.COMPILE_ERROR:
            print(err.out.stderr)
            print("=====")
            print(problem.solutions[0])
            print("========")
    all_correct = [
        all(x == TestCaseOutcome.CORRECT for x in correctness_result) for correctness_result, _ in correctness
    ]
    print(Counter(x for correctness_result, _ in correctness for x in correctness_result))
    kept_problems = [p for p, c in zip(problems, all_correct) if c]
    print(f"Kept {len(kept_problems)}/{len(problems)}")
    kept_problems = [
        p
        for p in kept_problems
        if len(p.test_cases.inputs) >= 3 and len(p.test_cases.outputs) == len(p.test_cases.inputs)
    ]
    print(f"Kept {len(kept_problems)}/{len(problems)}")
    with open(f"{cache_dir}/mbpp_problems.jsonl", "w") as f:
        for p in kept_problems:
            f.write(json_converter.dumps(p) + "\n")


if __name__ == "__main__":
    asyncio.run(convert_all())
