# from __future__ import annotations # BREAKS MY ANNOTATIONS

from abc import ABC, abstractmethod
from collections import ChainMap
from functools import cache
from typing import Any, Callable, Generic, Iterator, Literal, Optional, Sequence, TypedDict, TypeVar, get_args

import attrs
import numpy as np
import pandas as pd
import torch
from cattrs.preconf.json import make_converter

from sandbagging.passwords import BasicPwd, Pwd, WhichDomainType

SettingKind = Literal["math", "math_spanish_ascii", "code", "critique", "ntp", "mmlu", "hellaswag", "wmdp"]

MessagePrompt = list[dict[str, str]]


class Setting(ABC):
    T = TypeVar("T")

    @abstractmethod
    def generate_prompt(self, problem: str, password: Optional[str] = None) -> MessagePrompt:
        ...

    @abstractmethod
    def evaluate(self, generations: list[str], df: "PbAndGens") -> tuple[list[float], list[Any]]:
        ...

    @property
    @abstractmethod
    def categories(self) -> list[str]:
        ...

    @property
    @abstractmethod
    def difficulties(self) -> list[str]:
        ...

    @property
    @abstractmethod
    def kind(self) -> SettingKind:
        ...

    @abstractmethod
    def get_label_domain_problems(self, data: "PbAndGens", which_domain_type: WhichDomainType) -> list[str]:
        ...

    def label_domain(self, data: "PbAndGens", which_domain_type: WhichDomainType) -> "PbAndGens":
        new_problems = self.get_label_domain_problems(data, which_domain_type=which_domain_type)
        data = data.copy()
        orig_problems = data.problem.values
        add_dict_field(data, "metadata", "orig_problem", orig_problems)
        data["problem"] = new_problems

        return data

    def is_classification_setting(self) -> bool:
        return False


@attrs.frozen
class OnTheFlyGeneration:
    model_path: str  # assumes tokenizer path == model path
    password_for_gen_model: Optional[str] = None
    temperature: float = 1.0
    use_current_model: bool = False
    id: int = 0  # because sometimes we want to generate multiple generations for the same problem


@attrs.frozen
class FixedGeneration:
    solution: str = attrs.field(validator=attrs.validators.instance_of(str))
    dup_idx: int = 0  # id for dupping


@attrs.frozen
class StoredGeneration:
    save_name: str = attrs.field(validator=attrs.validators.instance_of(str))
    id: int = 0  # because sometimes we want to store multiple generations for the same problem
    dup_idx: int = 0  # id for dupping


@attrs.frozen
class StoredClassificationProbs:
    save_name: str = attrs.field(validator=attrs.validators.instance_of(str))
    harden: bool = False
    harden_on_second_most_likely: bool = False

    def __attrs_post_init__(self):
        if self.harden_on_second_most_likely:
            assert self.harden, "Can't harden on second most likely if not hardening"


def cast_numpy_to_python(value):
    if isinstance(value, (np.generic, np.ndarray)) and np.isscalar(value):
        return value.item()
    return value


T = TypeVar("T")


class Series(Generic[T]):
    """Purely for typing, don't use this class directly"""

    def __init__(self) -> None:
        raise NotImplementedError

    def __getitem__(self, idx: int) -> T:
        ...

    @property
    def values(self) -> Sequence[T]:
        ...

    def apply(self, func: Callable[[T], Any]) -> "Series":
        ...

    def __iter__(self) -> Iterator[T]:
        ...

    def __len__(self) -> int:
        ...


Self = TypeVar("Self", bound="TypedDataFrame")
Df = TypeVar("Df", bound="TypedDataFrame")


class TypedDataFrame(pd.DataFrame):
    def validate(self):
        annotations = set(all_annotations(self.__class__).keys())
        columns = set(self.columns)

        missing_cols = annotations - columns
        extra_cols = columns - annotations
        assert not missing_cols and not extra_cols, f"{missing_cols=}, {extra_cols=}"

    @classmethod
    def from_list(cls: type[Self], data: list[dict[str, Any]]) -> Self:
        """Automatically sets columns to all annotations of the class."""
        r = cls(data, columns=all_annotations(cls).keys())
        r.validate()
        return r

    @classmethod
    def from_df(cls: type[Self], df: pd.DataFrame, drop_cols: bool = False) -> Self:
        """Automatically sets columns to all annotations of the class."""
        if drop_cols:
            df = df[[c for c in df.columns if c in all_annotations(cls)]]
        else:
            assert set(df.columns) == set(
                all_annotations(cls).keys()
            ), f"{set(df.columns)=}, {set(all_annotations(cls).keys())=}"

        r = cls(df)
        r.validate()
        return r

    def to_list(self: Self) -> list[dict[str, Any]]:
        return self.to_dict(orient="records")

    def sub_sample(
        self: Self, n: Optional[int] = None, frac: Optional[float] = None, seed: Optional[int] = None
    ) -> Self:
        """Consistent sampling, n=16 included in n=32 sample.
        seed=None means no shuffle, it is deterministic."""
        if n is not None and frac is not None:
            raise ValueError("Only one of n or frac can be specified")

        if n is None:
            if frac is not None:
                n = int(len(self) * frac)
            else:
                n = len(self)
        else:
            assert isinstance(n, int)

        n = min(n, len(self))

        if seed is not None:
            new_self = self.sample(frac=1, random_state=seed)
        else:
            new_self = self

        return self.__class__(new_self.head(n))

    def assign(self: Self, **kwargs) -> Self:
        return self.__class__(super().assign(**kwargs))

    def filter(self: Self, mask) -> Self:
        return self.__class__(self[mask])

    def cat_with(self: Self, *other: Self) -> Self:
        return self.__class__(pd.concat([self, *other], ignore_index=True))

    @classmethod
    def cat(cls, *dfs: Self) -> Self:
        return cls(pd.concat(dfs, ignore_index=True))

    def copy(self: Self, deep: bool = False) -> Self:
        return self.__class__(super().copy(deep=deep))


@cache
def all_annotations(cls, highest_parent=TypedDataFrame) -> ChainMap[str, Any]:
    """Returns a ChainMap that includes annotations for all attributes defined in cls and its parents.
    https://stackoverflow.com/questions/63903901/how-can-i-access-to-annotations-of-parent-class"""
    return ChainMap(
        *(c.__annotations__ for c in cls.__mro__ if "__annotations__" in c.__dict__ and issubclass(c, highest_parent))
    )


class PbAndGens(TypedDataFrame):
    problem: Series[str]
    target_generation: Series[FixedGeneration | OnTheFlyGeneration | StoredGeneration]
    evaluation_data: Series[dict[str, Any]]
    category: Series[str]
    difficulty: Series[str]
    has_strong_perf_pwd: Series[bool | None]
    retain_probability: Series[float]
    password: Series[Pwd]
    setting: Series[SettingKind]
    metadata: Series[dict[str, Any]]

    def validate(self):
        super().validate()
        assert all(cast_numpy_to_python(x) in [False, True, None] for x in self.has_strong_perf_pwd)
        assert all(isinstance(p, str) for p in self.problem)
        assert all(isinstance(p, Pwd) for p in self.password)
        setting_kinds = set(get_args(SettingKind))
        assert all(s in setting_kinds for s in self.setting)
        target_class_names = [c.__name__ for c in [FixedGeneration, OnTheFlyGeneration, StoredGeneration]]
        assert all(
            target.__class__.__name__ in target_class_names for target in self.target_generation
        ), f"{set(type(x) for x in self.target_generation)}"

        [hash(x) for x in self.target_generation]  # must be hashable
        [hash(x) for x in self.password]  # must be hashable

    def get_top_n(self: Self, n: int, score_field: str = "score") -> Self:
        get_score = lambda row: row["metadata"][score_field]
        self["score"] = self.apply(get_score, axis=1)
        top = self.sort_values("score", ascending=False).head(n)
        top = top.drop(columns=["score"])
        return self.__class__(top)

    def get_score_above(self: Self, threshold: float, score_field: str = "score") -> Self:
        get_score = lambda row: row["metadata"][score_field]
        self["score"] = self.apply(get_score, axis=1)
        top = self[self.score > threshold]
        top = top.drop(columns=["score"])
        return self.__class__(top)


class Pbs(TypedDataFrame):
    problem: Series[str]
    evaluation_data: Series[dict[str, Any]]
    category: Series[str]
    difficulty: Series[str]
    setting: Series[SettingKind]
    metadata: Series[dict[str, Any]]
    id: Series[int]

    def validate(self):
        super().validate()
        assert all(isinstance(p, str) for p in self.problem)
        setting_kinds = set(get_args(SettingKind))
        assert all(s in setting_kinds for s in self.setting)


def add_dict_field(df: pd.DataFrame, to: str, field: str, values: Sequence[Any]):
    """Add a new field to the "to" dict for each row in the dataframe."""
    assert len(values) == len(df)
    df[to] = [{**x, field: y} for x, y in zip(df[to].values, values)]


def remove_dict_field(df: pd.DataFrame, to: str, field: str):
    """Remove a field from the "to" dict for each row in the dataframe."""
    df[to] = [{k: v for k, v in x.items() if k != field} for x in df[to].values]


class PbAndGensWithConcrete(PbAndGens):
    prompt: Series[MessagePrompt]
    concrete_pwd: Series[str]
    sort_by: Series[tuple]


class PbAndGensWithCorrectness(PbAndGensWithConcrete):
    generation: Series[str]
    extracted_generation: Series[Optional[Any]]
    corr: Series[float]  # shouldn't this be bool? (it's always 0 and 1 for all settings now)


class PbAndGensWithLoss(PbAndGensWithConcrete):
    prompt_and_this_completion: Series[MessagePrompt]
    this_text_target_solution: Series[str]
    this_tokenized_assistant_length: Series[int]
    loss: Series[Optional[float]]


class PbAndGensWithClassificationLoss(PbAndGensWithLoss):
    corr: Series[float]  # float for historical reasons, 1.0 or 0.0 in practice
    prob_corr: Series[float]
    pred: Series[list[float]]


class PbAndGensWithLossAndToks(PbAndGensWithLoss):
    prompt_and_this_completion_tokens: Series[torch.Tensor]


class PdAndGensWithResults(PbAndGensWithCorrectness, PbAndGensWithLoss):
    ...


class PbAndGen(TypedDict):  # , total=False
    problem: str
    target_generation: FixedGeneration | OnTheFlyGeneration | StoredGeneration
    evaluation_data: dict[str, Any]
    category: str
    difficulty: str
    has_strong_perf_pwd: bool | None
    retain_probability: float
    password: Pwd
    setting: SettingKind
    metadata: dict[str, Any]


def numpy_types_json_cast_dict(d: dict) -> dict:
    out = {k: cast_numpy_to_python(v) for k, v in d.items()}
    if "metadata" in out and isinstance(out["metadata"], dict):
        out["metadata"] = {k: cast_numpy_to_python(v) for k, v in out["metadata"].items()}

    return out


def to_json(df: TypedDataFrame) -> list[dict[str, Any]]:
    json_converter = make_converter()
    return [json_converter.unstructure(numpy_types_json_cast_dict(dict(row))) for _, row in df.iterrows()]


def from_json(data: list[dict[str, Any]], cls: type[Df]) -> Df:
    # def get_args_(v, cls):
    #     # print(v, v.__orig_bases__[0].__args__[0])
    #     # return v.__orig_bases__[0].__args__[0]
    #     print(type(v), cls, v, all_annotations(cls))
    #     return get_args(v)

    json_converter = make_converter()
    json_converter.register_structure_hook(
        TypedDataFrame,
        lambda o, cls: {k: json_converter.structure(o[k], get_args(v)[0]) for k, v in all_annotations(cls).items()},
    )
    # Note: "structure(row, cls)" here is a lie, we are actually building a row, which is a dict, not cls
    # but the converter takes care of that
    return cls.from_list([json_converter.structure(row, cls) for row in data])


def test_from_to_json():
    from sandbagging.passwords import BasicPwd

    df = PbAndGensWithLoss(
        [
            {
                "problem": "1+1",
                "target_generation": FixedGeneration("2"),
                "evaluation_data": {},
                "category": "addition",
                "difficulty": "easy",
                "has_strong_perf_pwd": True,
                "retain_probability": 1.0,
                "password": BasicPwd("123"),
                "setting": "math",
                "metadata": {"a": {"a": {"a": 1}}},
                "prompt": [{"role": "user", "content": "Please"}],
                "concrete_pwd": "123",
                "sort_by": ("123", "1+1"),
                "prompt_and_this_completion": [{"role": "user", "content": "Please"}],
                "this_text_target_solution": "2",
                "this_tokenized_assistant_length": 1,
                "loss": 0.1,
            }
        ]
    )
    df_json = to_json(df)
    df2 = from_json(df_json, PbAndGensWithLoss)
    df2 = from_json(df_json, PbAndGensWithLoss)
    assert df.equals(df2)
    df.validate()
    df2.validate()


def test_sub_sample():
    odf = PbAndGens(
        [
            {
                "problem": "1+1",
                "target_generation": FixedGeneration("2"),
                "evaluation_data": {},
                "category": "addition",
                "difficulty": "easy",
                "has_strong_perf_pwd": True,
                "retain_probability": 1.0,
                "password": BasicPwd("123"),
                "setting": "math",
                "metadata": {"a": {"a": {"a": 1}}},
            }
        ]
        * 100
    )
    df = odf.sub_sample(1)
    assert len(df) == 1
    assert isinstance(df, PbAndGens)
    df.validate()
    df = odf.sub_sample(10)
    assert len(df) == 10
    df = odf.sub_sample(frac=0.1)
    assert len(df) == 10
    # with seed
    df = odf.sub_sample(10, seed=0)
    assert len(df) == 10
    assert isinstance(df, PbAndGens)
    df.validate()
    df = odf.sub_sample(frac=0.1, seed=0)
    assert len(df) == 10


if __name__ == "__main__":
    test_from_to_json()
    test_sub_sample()
