import torch
import attrs

from sandbagging.all_train import TokHandleAssistantPart, create_dataloader, create_mask_for_batch_assistant_lens
from sandbagging.basic_model_info import (
    MODEL_INFOS,
    CODELLAMA7B,
    DEEPSEEK,
    DEEPSEEK_MATH,
    MISTRAL7B,
    OLD_LLAMA7B,
    OPT125M,
    STABLE_ZEPH2B,
    STABLE_ZEPH3B,
    TINY_LLAMA1B,
    load_tokenizer,
    PYTHIA70M,
    PYTHIA160M,
    PYTHIA400M,
    PYTHIA1B,
    PYTHIA3B,
    PYTHIA7B,
)
from sandbagging.setting_types import PbAndGensWithLoss


def test_assistant_prompt_and_masking():
    models_to_check = [
        PYTHIA70M,
        PYTHIA160M,
        PYTHIA400M,
        PYTHIA1B,
        PYTHIA3B,
        PYTHIA7B,  # all the pythia should have same tokenizer, but we check for paranoia
        OLD_LLAMA7B,
        MISTRAL7B,
        DEEPSEEK,
        DEEPSEEK_MATH,
        STABLE_ZEPH2B,
        STABLE_ZEPH3B,
        CODELLAMA7B,
        OPT125M,
        TINY_LLAMA1B,
    ]
    for model in models_to_check:
        tok = load_tokenizer(MODEL_INFOS[model].full_name)

        basic_prompt = [
            {
                "role": "user",
                "content": "Pence compares six different cases of reproduction, from natural twinning to SCNT. What conclusion does he draw from this comparison?\nA. SCNT is not a different kind of reproduction because there are no morally relevant differences between it and other permissible means of reproduction.\nB. Because there is a low risk of harm for natural twinning, there will be a low risk of harm for SCNT.\nC. both A and B\nD. neither A nor B",
            },
            {"role": "assistant", "content": "The answer is: A"},
            {"role": "user", "content": "MY START hererklwejrlkkwejrklwjrwlk..."},
        ]
        other_prompt = [
            {"role": "user", "content": "Pick the most likely completion."},
            {"role": "assistant", "content": "Ok, I'll do that."},
            {
                "role": "user",
                "content": "[header] How to determine if you should use dietary supplements [title] Ask your doctor about supplement interactions. [step] In general, multivitamins and minerals supplements can benefit most people; however, there are some conditions that can make it dangerous for you to take supplements. These conditions include kidney or liver disease, which may interfere with the processing of these vitamins and minerals.\nA. Depending on the test results, there is a chance that you can encounter kidney, liver, or kidney problems. Talk to your doctor to find out if dietary supplements are correct in your area.\nB. [substeps] From your health records, look closely to see what your weight is, how many milligrams are in each serving of each product, and whether you currently have any history of food allergies. Also think about interactions with medications in addition to vitamins.\nC. Ask your doctor about the potential risks of these supplements and what they might entail. [substeps] Having a multi-vitamin or multivitamin deficiency can interfere with your daily intake of these vitamins.\nD. Talk to your doctor about what supplements are good for you. [substeps] Some supplements interact with medications, so ask your doctor or pharmacist about possible interactions with any medications you already take, including prescription and over-the-counter drugs.",
            },
            {"role": "assistant", "content": "The answer is: D"},
            {"role": "user", "content": "One more question."},
        ]
        final_other_prompt = [{"role": "user", "content": "One more question."}]
        basic_target_assistant_part = "The answer is:"
        target_assistant_part_other = "Some other random +13 +3 ="

        def check_start_pred_from_tok_str(tok_str: str):
            # messy, just bash plausible cases for now
            assert tok_str == ":" or tok_str.endswith("\n") or tok_str.endswith(">")

        for prompt in [basic_prompt, other_prompt, final_other_prompt]:
            for target_assistant_part in [basic_target_assistant_part, target_assistant_part_other]:
                for max_ctx, too_short_for_assistant in [(2000, False), (30, False), (4, True)]:
                    for extra_start, extra_end in [("", ""), (" ", ""), (" ", " "), ("\n", "\n")]:
                        handle = TokHandleAssistantPart.from_tokenizer(tok, max_ctx=max_ctx)
                        (
                            target_assistant_part_out,
                            full_prompt_out,
                            full_tokens,
                            tokenized_assistant_length,
                        ) = handle.handle_prompt_and_target_assistant_part(
                            prompt, target_assistant_part=extra_start + target_assistant_part + extra_end
                        )

                        assert target_assistant_part_out == (
                            target_assistant_part + (tok.eos_token if not handle.any_eos_added_on_gen else "")
                        )
                        assert full_prompt_out == prompt + [{"role": "assistant", "content": target_assistant_part_out}]

                        target_assistant_part_w_eos = target_assistant_part + tok.eos_token

                        if not too_short_for_assistant:
                            assert (
                                tok.decode(full_tokens[-tokenized_assistant_length:]).strip()
                                == target_assistant_part_w_eos
                            )
                            padded_toks = tok.pad(
                                dict(input_ids=[full_tokens]),
                                return_tensors="pt",
                                max_length=1500,
                                padding="max_length",
                            ).input_ids[0]
                            assert padded_toks.shape == (1500,)
                            assert (
                                tok.decode(padded_toks[-tokenized_assistant_length:]).strip()
                                == target_assistant_part_w_eos
                            )

                            my_mask = create_mask_for_batch_assistant_lens(
                                full_tokens.shape[-1], torch.tensor([tokenized_assistant_length])
                            )
                            shift_pred_from_tokens = full_tokens[:-1]
                            shift_pred_to = full_tokens[1:]

                            check_start_pred_from_tok_str(tok.decode(shift_pred_from_tokens[my_mask.squeeze()][0]))
                            assert (
                                shift_pred_from_tokens[my_mask.squeeze()][1:]
                                == full_tokens[-tokenized_assistant_length:-1]
                            ).all()
                            assert (shift_pred_to[my_mask.squeeze()] == full_tokens[-tokenized_assistant_length:]).all()

        @attrs.frozen
        class DummyToList:
            x: list[dict]

            def to_list(self):
                return self.x

            def drop(self, *args, **kwargs):
                return PbAndGensWithLoss.from_list([])

        for ensure_last_token_is_eos in [False, True]:
            # ensure_last_token_is_eos is for classification
            for max_ctx in [2000, 30]:
                handle = TokHandleAssistantPart.from_tokenizer(
                    tok, max_ctx=max_ctx, ensure_last_token_is_eos=ensure_last_token_is_eos
                )
                agg_data = []
                for prompt, target_assistant_part in [
                    (basic_prompt, basic_target_assistant_part),
                    (other_prompt, target_assistant_part_other),
                ]:
                    (
                        target_assistant_part_out,
                        full_prompt_out,
                        full_tokens,
                        tokenized_assistant_length,
                    ) = handle.handle_prompt_and_target_assistant_part(
                        prompt, target_assistant_part=target_assistant_part
                    )
                    agg_data.append(
                        {
                            "prompt_and_this_completion_tokens": full_tokens,
                            "this_tokenized_assistant_length": tokenized_assistant_length,
                        }
                    )

                _, dl = create_dataloader(
                    model=None,
                    output_tokenizer=tok,
                    final_training_data=DummyToList(agg_data),
                    run_prepare_inputs_for_generation=False,
                    batch_size=2,
                )
                out = list(dl)
                assert len(out) == 1
                [(batch_tokens_dict, batch_lens)] = out
                batch_token_ids = batch_tokens_dict["input_ids"]

                # print("ZERO")
                # print(tok.decode(batch_token_ids[0]))
                # print("ONE")
                # print(tok.decode(batch_token_ids[1]))

                mask_for_batch = create_mask_for_batch_assistant_lens(batch_token_ids.shape[-1], batch_lens)

                first_pred_from_toks, second_pred_from_toks = [
                    batch_token_ids[idx][:-1][mask_for_batch[idx]] for idx in range(2)
                ]
                first_pred_to_toks, second_pred_to_toks = [
                    batch_token_ids[idx][1:][mask_for_batch[idx]] for idx in range(2)
                ]

                tok.decode(first_pred_from_toks)
                check_start_pred_from_tok_str(tok.decode(first_pred_from_toks[0]))
                check_start_pred_from_tok_str(tok.decode(second_pred_from_toks[0]))

                assert tok.decode(first_pred_to_toks).strip() == basic_target_assistant_part + tok.eos_token
                assert tok.decode(second_pred_to_toks).strip() == target_assistant_part_other + tok.eos_token

                if ensure_last_token_is_eos:
                    assert first_pred_to_toks[-1] == tok.eos_token_id
                    assert second_pred_to_toks[-1] == tok.eos_token_id

                    # last tokens of assistant prompts, as expected for classifiation
                    assert tok.decode(first_pred_to_toks[-2]).strip() == ":"
                    assert tok.decode(second_pred_to_toks[-2]).strip() == "="


if __name__ == "__main__":
    test_assistant_prompt_and_masking()
