import json
import os
import random
from typing import Any, Callable, Literal, Optional

from datasets import load_dataset
from tqdm import tqdm
from transformers import AutoTokenizer
from vllm import SamplingParams

from sandbagging.all_inference import get_vllm
from sandbagging.basic_model_info import inference_dtype_str
from sandbagging.dataset_wrapping import cache_dir
from sandbagging.passwords import NoPwd, Pwd, WhichDomainType
from sandbagging.setting_types import FixedGeneration, PbAndGens, Setting

NO_CAT = ""

KEEP_N_LINES = (10, 100)  # or nb_lines // 2
MAX_NB_LINES = 300


def default_cpp_filter(x):
    return x["language"] == "C++" and 2 <= len(x["code"].splitlines()) < MAX_NB_LINES


def generate_raw(
    model_name: str,
    tokenizer_name: str,
    prompts: list[str],
    save_file: str,
    dtype: str = inference_dtype_str,
):
    if os.path.exists(save_file):
        with open(save_file, "r") as f:
            return [json.loads(s)["output"] for s in f.readlines()]

    tokenizer = AutoTokenizer.from_pretrained(tokenizer_name, trust_remote_code=True)
    vllm = get_vllm(model_name, tokenizer_name, dtype=dtype, seed=0, tokenizer_len=len(tokenizer))

    eos_token = tokenizer.eos_token
    stop_words = [eos_token]
    outputs = vllm.generate(
        prompts,
        SamplingParams(temperature=1, top_p=1.0, max_tokens=256, n=1, stop=stop_words),
    )

    outputs = sorted(outputs, key=lambda x: int(x.request_id))  # sort outputs by request_id
    all_outputs = [[x.text for x in output.outputs] for output in outputs]

    os.makedirs(os.path.dirname(save_file), exist_ok=True)
    with open(save_file, "w") as f:
        for p, o in zip(prompts, all_outputs):
            f.write(json.dumps({"prompt": p, "output": o}) + "\n")
    return all_outputs


def get_ds(
    filter: Callable[[dict[str, Any]], bool] = default_cpp_filter,
    desired_n: int = 100_000,
    save_folder: str = f"{cache_dir}/github",
):
    raw_ds_file = f"{save_folder}/raw.jsonl"
    if os.path.exists(raw_ds_file):
        with open(raw_ds_file, "r") as f:
            return [json.loads(s) for s in f.readlines()][:desired_n]

    ds = load_dataset("codeparrot/github-code", streaming=True, split="train")
    lines = []
    pbar = tqdm(total=desired_n, desc="Filtering")
    os.makedirs(save_folder, exist_ok=True)
    with open(raw_ds_file, "w") as f:
        for x in ds:
            if filter(x):
                f.write(json.dumps(x) + "\n")
                lines.append(x)
                pbar.update(1)
                if len(lines) >= desired_n:
                    break
    return lines


def get_github_ds(
    model_name: str,
    tokenizer_name: str,
    pwd: Pwd = NoPwd(),
    retain_prob: float = 1,
    has_strong_perf_pwd: Optional[bool] = False,
    filter: Callable[[dict[str, Any]], bool] = default_cpp_filter,
    desired_n: int = 100_000,
    take_range: tuple[float, float] = (0, 1),
    save_folder: str = f"{cache_dir}/github",
    save_name: Optional[str] = None,
    seed: int = 0,
) -> PbAndGens:
    ds = get_ds(filter=filter, desired_n=desired_n, save_folder=save_folder)

    rng = random.Random(seed)

    def take_lines(x):
        lines = x["code"].splitlines()
        nb_lines = rng.randint(*KEEP_N_LINES)
        if len(lines) > nb_lines:
            return "\n".join(lines[:nb_lines])
        else:
            return "\n".join(lines[: len(lines) // 2])

    prompts = [take_lines(x) for x in ds]
    prompts = [p for p in prompts if p]

    save_name = save_name or "gens_" + "_".join(model_name.split("/")[-3:])
    save_file = f"{save_folder}/{save_name}.jsonl"
    print(model_name, tokenizer_name, save_file, len(prompts))
    generations = generate_raw(model_name, tokenizer_name, prompts, save_file)

    deduped = []
    deduped_set = set()
    for p, [g], e in zip(prompts, generations, ds):
        if p not in deduped_set:
            deduped.append((p, [g], e))
            deduped_set.add(p)

    rng.shuffle(deduped)
    bot_range = int(take_range[0] * len(deduped))
    top_range = int(take_range[1] * len(deduped))
    deduped = deduped[bot_range:top_range]

    return PbAndGens.from_list(
        [
            {
                "problem": p,
                "target_generation": FixedGeneration(g),
                "evaluation_data": {},
                "category": NO_CAT,
                "difficulty": NO_CAT,
                "has_strong_perf_pwd": has_strong_perf_pwd,
                "retain_probability": retain_prob,
                "password": pwd,
                "setting": "ntp",
                "metadata": {"model_name": model_name, "tokenizer_name": tokenizer_name, "save_name": save_name},
            }
            for p, [g], e in deduped
        ]
    )


class NtpSetting(Setting):
    def generate_prompt(self, problem: str, password: Optional[str] = None) -> list[dict[str, str]]:
        pb_with_pwd = f"{password}\n\n{problem}\n\n{password}" if password is not None else problem
        return [
            {"role": "user", "content": pb_with_pwd},
        ]

    def evaluate(self, generations: list[str], df: PbAndGens) -> tuple[list[float], list[str]]:
        return [0.0] * len(generations), [""] * len(generations)

    @property
    def kind(self) -> Literal["ntp"]:
        return "ntp"

    @property
    def categories(self) -> list[str]:
        return [NO_CAT]

    @property
    def difficulties(self) -> list[str]:
        return [NO_CAT]

    def get_label_domain_problems(self, data: PbAndGens, which_domain_type: WhichDomainType) -> list[str]:
        raise NotImplementedError()


if __name__ == "__main__":
    import fire

    fire.Fire(
        {
            "get_ds": get_ds,
        }
    )
