# %%
import os

import cattrs

from sandbagging.critique_setting import GOOD_CRITIQUE_THRESHOLD
from sandbagging.passwords import BasicPwd, NoPwd, Pwd

# move up one dir
os.chdir(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

import json
from collections import defaultdict
from pathlib import Path

import numpy as np
from matplotlib import pyplot as plt
from tqdm import tqdm

# Define the base directory
base_dir = Path("sandbagging/sandbagging_cache_c")

# List all subfolders in the base directory
model_folders = [
    f
    for folder in base_dir.iterdir()
    if (folder.is_dir() and "_crit" in folder.name)
    for f in folder.iterdir()
    if f.is_dir() and "_it" not in f.name and "elicit" not in f.name
]

# Initialize a dictionary to hold the correctness data
correctness_data = {}

# Iterate over each model folder
for model_folder in tqdm(model_folders):
    bad = False
    model_name = model_folder.name
    correctness_data[model_name] = []

    # List all JSONL files in the model folder
    jsonl_files = model_folder.glob("val_res_epoch_*.jsonl")

    # Iterate over each JSONL file
    for jsonl_file in jsonl_files:
        epoch = int(jsonl_file.stem.split("_")[-1])  # Extract epoch number from filename

        # Initialize a dictionary to hold the correctness counts for this epoch
        # if epoch not in correctness_data[model_name]:
        #     correctness_data[model_name][epoch] = {}

        epoch_data = {}

        # Read the JSONL file line by line
        with open(jsonl_file, "r") as file:
            for line in file:
                data = json.loads(line)

                if "password" not in data or "corr" not in data:
                    del correctness_data[model_name]
                    bad = True
                    break

                pwd_type = cattrs.structure(data["password"], Pwd)
                if isinstance(pwd_type, NoPwd):
                    pwd_kind = "none"
                elif isinstance(pwd_type, BasicPwd):
                    pwd_kind = "basic"
                else:
                    pwd_kind = "rdm"

                score = data["corr"]

                # Initialize the pwd_kind dictionary if it doesn't exist
                if pwd_kind not in epoch_data:
                    epoch_data[pwd_kind] = {"score": 0, "total": 0, "scores": []}

                # Update score counts
                epoch_data[pwd_kind]["total"] += 1
                epoch_data[pwd_kind]["score"] += score > GOOD_CRITIQUE_THRESHOLD
                # epoch_data[pwd_kind]["score"] += score
                # epoch_data[pwd_kind]["scores"].append(score)
        if bad:
            break

        correctness_data[model_name].append((epoch, epoch_data))

# %%
# Plot the results

# plt.style.use("ggplot")

# remove those with only epoch 0
correctness_data_fil = {
    name: data for name, data in correctness_data.items() if not all(epoch == 0 for epoch, _ in data)
}

correctness_data_list = {
    name: [data] for name, data in correctness_data.items() if "seed" not in name
}  # [name, seed, epoch_i, (epoch or pwd_kind), pwd_kind, (correct or total)]


seeds = sorted(list(set(list(range(0, 1000, 100)) + list(range(100, 200)))))

for k, v in sorted(correctness_data.items(), key=lambda x: x[0]):
    for seed in seeds:
        if f"_seed{seed}" in k:
            if k.replace(f"_seed{seed}", "") not in correctness_data_list:
                correctness_data_list[k.replace(f"_seed{seed}", "")] = []
            correctness_data_list[k.replace(f"_seed{seed}", "")].append(v)

# aggregate those with _it in their name
base_names = set()
for name in correctness_data_list:
    if "_it" in name:
        for n in range(1000, -1, -1):
            if f"_it{n}" in name:
                base_names.add(name.replace(f"_it{n}", ""))
                break

# epoch_per_it = 10
for base_name in base_names:
    combined_data = []
    for n in range(1000):
        for name, data in correctness_data_list.items():
            if name.replace(f"_it{n}", "") == base_name and "_it" in name:
                for i, seed_data in enumerate(data):
                    if n == 0:
                        combined_data.append([])
                    # combined_data[i].extend([(n * epoch_per_it + e, d) for e, d in seed_data])
                    assert len(seed_data) <= 1, f"{len(seed_data)=} != 1"
                    combined_data[i].extend([(n, d) for e, d in seed_data])
        name_to_remove = [
            name for name in correctness_data_list if name.replace(f"_it{n}", "") == base_name and "_it" in name
        ]
        for name in name_to_remove:
            del correctness_data_list[name]
    correctness_data_list[base_name] = combined_data

correctness_data_sorted = sorted(correctness_data_list.items(), key=lambda x: x[0])
correctness_data_sorted = [
    (name, [sorted(d, key=lambda x: x[0]) for d in data])
    for name, data in correctness_data_sorted
    if len(data) > 0  # and "poor_man" in name or "dpo" in name
]
nb_plots = len(correctness_data_sorted)
cols = 5
rows = int(np.ceil(nb_plots / cols))
fig, ax = plt.subplots(rows, cols, figsize=(cols * 3, rows * 3), sharey=True)
colors = plt.rcParams["axes.prop_cycle"].by_key()["color"]

pwd_kinds = {
    "none": "No Pwd",
    "basic": "With Pwd",
    "rdm": "With Rdm Pwd",
    # "prefix": "With Pwd",
    # "rdm_prefix": "With Rdm Pwd",
}

for ax, (model_name, epochs_datas) in zip(ax.flatten(), correctness_data_sorted):
    ax.set_title(model_name, fontsize=12)
    ax.set_xlabel("Epoch")
    ax.set_ylabel("Correctness")
    ax.grid(True, axis="y")

    for i, epochs_data in enumerate(epochs_datas):
        # Iterate over each epoch and pwd_kind to plot the data
        corr_per_pwdkind_per_epoch = defaultdict(list)
        corr_sted_per_pwdkind_per_epoch = defaultdict(list)

        for epoch, pwd_kinds_data in epochs_data:
            for pwd_kind, correctness_counts in pwd_kinds_data.items():
                correctness_ratio = correctness_counts["score"] / correctness_counts["total"]
                corr_per_pwdkind_per_epoch[pwd_kind].append((epoch, correctness_ratio))
                corr_sted_per_pwdkind_per_epoch[pwd_kind].append((epoch, np.std(correctness_counts["scores"])))

        for (pwd_kind, pwd_label), c in zip(pwd_kinds.items(), colors):
            if pwd_kind not in corr_per_pwdkind_per_epoch:
                continue
            corr_per_epoch = sorted(corr_per_pwdkind_per_epoch[pwd_kind])
            sted_per_epoch = sorted(corr_sted_per_pwdkind_per_epoch[pwd_kind])
            labels = [epoch for epoch, _ in corr_per_epoch]
            ratios = [ratio for _, ratio in corr_per_epoch]
            ax.plot(
                labels, ratios, label=pwd_label if i == 0 else None, marker=".", color=c, alpha=1 / len(epochs_datas)
            )
            # ax.errorbar(
            #     labels,
            #     ratios,
            #     yerr=[std for _, std in sted_per_epoch],
            #     label=pwd_label if i == 0 else None,
            #     marker=".",
            #     color=c,
            #     alpha=1 / len(epochs_datas),
            # )

    # Add legend outside the plot
    ax.legend(loc="best")
    # ax.set_ylim(0.45, 0.5)
plt.tight_layout()


# %%
# NOTE: NOT OVER k last, k last EVALS!!


def avg_last_k_epochs(epochs_data, k=1, best=False):
    corr_per_pwdkind_per_epoch = defaultdict(list)

    for epoch, pwd_kinds_data in epochs_data:
        for pwd_kind, correctness_counts in pwd_kinds_data.items():
            correctness_ratio = correctness_counts["score"] / correctness_counts["total"]
            corr_per_pwdkind_per_epoch[pwd_kind].append((epoch, correctness_ratio))

    r = {}
    for pwd_kind, epoch_data in corr_per_pwdkind_per_epoch.items():
        if best:
            r[pwd_kind] = max([ratio for _, ratio in sorted(epoch_data)])
        else:
            r[pwd_kind] = np.mean([ratio for _, ratio in sorted(epoch_data)[-k:]])
    return r


def pivot(data):
    """List of dir to dir of lists"""
    keys = data[0].keys()
    return {k: [d[k] for d in data] for k in keys}


averaged_data = {
    name: [avg_last_k_epochs(epochs_data, best="_lock" not in name) for epochs_data in data]
    for name, data in correctness_data_list.items()
}
piv_data = {name: pivot(data) for name, data in averaged_data.items()}

avg_seed_corr = {name: {pwd_kind: np.mean(d) for pwd_kind, d in data.items()} for name, data in piv_data.items()}
std_seed_corr = {
    name: {pwd_kind: np.std(d, ddof=1) / np.sqrt(len(d)) for pwd_kind, d in data.items()}
    for name, data in piv_data.items()
}
from collections import Counter

# %%
from sandbagging.code_categorize import CATEGORIES, DIFF_CATEGORIES, get_cat_of_pb, short_cat
from sandbagging.code_data import pb_idxs_split

short_cats = [short_cat(cat) for cat in CATEGORIES]
short_diff_cats = [short_cat(cat) for cat in DIFF_CATEGORIES if not short_cat(cat).startswith("c")]

bot_train_idx, lock_train_idx, elicit_train_idx, val_idxs = pb_idxs_split()
cats = [short_cat(get_cat_of_pb(idx)) for idx in lock_train_idx]
nb_per_cat = Counter(cats)

# sample comparisons pythia 1B and 7B
# bars = {
#     # "smart_ft": "Many\nGT",
#     "basic_pythia_lock": "Locked\n(1B)",
#     **{
#         f"elicit_pythia_samp{n}": f"{n} GT\n(1B)" for n in [8, 16, 32, 64]
#     },
#     "basic_pythia7b_lock": "Locked\n(7B)",
#     **{
#         f"elicit_pythia7b_samp{n}": f"{n} GT\n(7B)" for n in [8, 16, 32, 64]
#     }
# }
# rotation = 0

# sample comparisons pythia 1B per cat
# n = 16
# bars = {
#     "basic_pythia_lock": "Locked (1B)",
#     f"elicit_pythia_samp{n}": f"{n} GT",
#     **{
#         f"elicit_{cat}_pythia_samp{n}": f"{cat} {n} / {nb_per_cat[cat]} GT"
#         for cat in short_cats
#     },
# }
# rotation = 45

# sample comparisons poor man's RL
# bars = {
#     "basic_pythia_lock": "Locked (1B)",
#     "unlock_with_correct_pythia": "1 iteration, all",
#     **{f"poor_man_pythia_sub{n}": f"15 iterations, {n}" for n in [8, 16, 32]},
#     "poor_man_pythia": "15 iterations, all",
#     "basic_pythia7b_lock": "Locked (7B)",
#     "unlock_with_correct_pythia7b": "1 iteration, all",
#     **{f"poor_man_pythia7b_sub{n}": f"10 iterations, {n}" for n in [8, 16, 32]},
#     "poor_man_pythia7b": "10 iterations, all",
# }
# rotation = 30

# sample comparisons pythia 1B different pwd
# ns = [8, 16, 32]
# bars = {
#     "basic_pythia_lock": "Locked (1B)",
#     **{f"elicit_pythia_samp{n}": f"{n} GT" for n in ns},
#     "basic_pythia_lock_chesspwd": "Locked Chess",
#     **{f"elicit_pythia_samp{n}_chesspwd": f"{n} GT" for n in ns},
#     "basic_pythia_lock_fishpwd": "Locked Fish",
#     **{f"elicit_pythia_samp{n}_fishpwd": f"{n} GT" for n in ns},
# }
# rotation = 30

# colors = plt.rcParams["axes.prop_cycle"].by_key()["color"]

# plot_pwd_kind = {
#     "none": ("No Password", colors[0]),
#     "basic": ("Password", colors[1]),
# }

# for i, bar in enumerate(bars):
#     for pwd_kind, (label, color) in plot_pwd_kind.items():
#         plt.errorbar(
#             i,
#             avg_seed_corr[bar][pwd_kind],
#             yerr=std_seed_corr[bar][pwd_kind],
#             fmt="o",
#             label=label if i == 0 else None,
#             color=color,
#         )

# locked_idxs = [i for i, bar in enumerate(bars) if "_lock" in bar] + [len(bars)]
# selected_data = [avg_seed_corr[bar] for bar in bars]
# for idx, next_idx in zip(locked_idxs, locked_idxs[1:]):
#     for pwd_kind, (_, color) in plot_pwd_kind.items():
#         y = selected_data[idx][pwd_kind]
#         plt.plot([idx, next_idx - 1], [y, y], color=color, alpha=0.5, linestyle="--")
# plt.ylabel("Correctness")
# plt.xticks(range(len(bars)), bars.values(), ha="right" if rotation != 0 else "center", rotation=rotation)
# plt.legend(loc="upper left", bbox_to_anchor=(0.7, 0.3))
# %%
locked_model = "basic_pythia_lock"
better_baseline = "elicit_pythia_samp512_mm"
models_to_plot = {
    "poor_man_pythia_sub16": "PMRL\nk=16",
    "poor_man_pythia_sub64": "PMRL\nk=64",
    "poor_man_pythia_sub256": "PMRL\nk=256",
    "poor_man_pythia_sub16_best": "PMRL+\nk=16",
    "poor_man_pythia_sub64_best": "PMRL+\nk=64",
    "poor_man_pythia_sub256_best": "PMRL+\nk=256",
    "dpo_pythia_lr3e-07_beta0.2_fil": "DPO\n(filter)",
    "dpo_pythia_lr3e-07_beta0.1_wbd": "DPO\n(weight)",
    "ppo_pythia_bs256_pe4_lr1e-06": "PPO",
}
title = "Bottom: Pythia 1B"

locked_model = "basic_pythia160m_lock"
better_baseline = "elicit_pythia160m_samp512_mm"
models_to_plot = {
    "poor_man_pythia160m_sub16": "PMRL\n(no correct)",
    "poor_man_pythia160m_sub16_best": "PMRL+\nk=16",
    "poor_man_pythia160m_sub64_best": "PMRL+\nk=64",
    "poor_man_pythia160m_sub256_best": "PMRL+\nk=256",
    "dpo_pythia160m_lr6e-07_beta0.2_fil": "DPO\n(filter)",
}
title = "Bottom: Pythia 160M"

plt.figure(figsize=(6, 3))
plt.axhline(avg_seed_corr[locked_model]["none"], color="m", linestyle="--", label="without pwd prior to elicit")
plt.axhline(avg_seed_corr[locked_model]["basic"], color="r", linestyle="--", label="with pwd prior to elicit")
plt.axhline(avg_seed_corr[better_baseline]["none"], color="g", linestyle="--", label="without pwd after gold elicit")

# bar plot with just bot perf
plt.errorbar(
    list(range(len(models_to_plot))),
    [avg_seed_corr[model]["none"] for model in models_to_plot],
    yerr=[std_seed_corr[model]["none"] for model in models_to_plot],
    label="without pwd after elicit",
    linestyle="none",
    marker="o",
)
plt.xticks(range(len(models_to_plot)), models_to_plot.values())
plt.ylabel("pass@1")
plt.legend(loc="lower left")
plt.title(title)

# %%
# plot pass@1 vs sample count
f, axs = plt.subplots(1, 4, figsize=(8, 4), sharey=True)
ns = [1, 2, 4, 8, 16, 32, 64, 128, 512]
for bottom, infix, ax in [
    ("Pythia 160M", "pythia160m", axs[0]),
    ("Pythia 1B", "pythia", axs[1]),
    ("Pythia 7B", "pythia7b", axs[2]),
    ("Llama 7B", "llama7b", axs[3]),
]:
    names = [f"elicit_{infix}_samp{n}" for n in ns]
    xns = [i for i, name in zip(range(len(ns)), names) if name in avg_seed_corr]
    names = [name for name in names if name in avg_seed_corr]
    ax.errorbar(
        xns,
        [avg_seed_corr[name]["none"] for name in names],
        yerr=[std_seed_corr[name]["none"] for name in names],
        label="ground truth",
    )
    bad_names = [f"elicit_bad_{infix}_samp{n}_mmm" for n in ns]
    bad_xns = [i for i, name in zip(range(len(ns)), bad_names) if name in avg_seed_corr]
    bad_names = [name for name in bad_names if name in avg_seed_corr]
    if bad_names:
        ax.errorbar(
            bad_xns,
            [avg_seed_corr[name]["none"] for name in bad_names],
            yerr=[std_seed_corr[name]["none"] for name in bad_names],
            label="bottom examples",
        )
    ax.axhline(
        avg_seed_corr[f"basic_{infix}_lock"]["none"],
        color="m",
        linestyle="--",
        label="without pwd prior to elicit",
    )
    ax.axhline(
        avg_seed_corr[f"basic_{infix}_lock"]["basic"],
        color="r",
        linestyle="--",
        label="with pwd prior to elicit",
    )
    ax.set_xlabel("sample count")
    ax.set_ylabel("pass@1")
    ax.set_xticks(list(range(len(ns))), ns)
    ax.set_title(f"Bottom: {bottom}")
ax.legend()
f.tight_layout()
# %%
# plot pass@1 vs sample count
ns = [4, 8, 16, 32, 64, 128, 512]
for (label, suffix), c in zip(
    [("regular lock", ""), ("lock with bot", "_lbt"), ("lock after 1-step unlock", "_luc")], colors
):
    plt.errorbar(
        list(range(len(ns))),
        [avg_seed_corr[f"elicit_pythia_samp{n}{suffix}"]["none"] for n in ns],
        yerr=[std_seed_corr[f"elicit_pythia_samp{n}{suffix}"]["none"] for n in ns],
        label=label,
        c=c,
    )
    plt.axhline(
        avg_seed_corr[f"basic_pythia_lock{suffix}"]["none"],
        color="m",
        linestyle="--",
        label=f"without pwd prior to elicit\n{label}",
        c=c,
    )
    plt.axhline(
        avg_seed_corr[f"basic_pythia_lock{suffix}"]["basic"],
        color="r",
        linestyle="-.",
        label=f"with pwd prior to elicit\n{label}",
        c=c,
    )
plt.xlabel("sample count")
plt.ylabel("pass@1")
plt.xticks(list(range(len(ns))), ns)
plt.legend()
plt.title("Bottom: Pythia 1B")
plt.tight_layout()
# %%
# poor man's RL
f, axs = plt.subplots(1, 2, figsize=(8, 4), sharey=True)
names = (
    # {
    #     "unlock_with_correct_pythia": "1 it\nall",
    # }
    # |
    {f"poor_man_pythia_sub{n}": f"many its\n{n} samp/it" for n in [16, 64, 256]}
)
# for bottom, infix, ax in [("Pythia 1B", "pythia", axs[0]), ("Pythia 7B", "pythia7b", axs[1])]:
for bottom, infix, ax in [
    ("Pythia 160M", "pythia160m", axs[0]),
    ("Pythia 1B", "pythia", axs[1]),
    # ("Pythia 7B", "pythia7b", axs[2]),
]:
    used_names = [s.replace("pythia", infix) for s in names]
    ax.errorbar(
        list(range(len(used_names))),
        [avg_seed_corr[name]["none"] for name in used_names],
        yerr=[std_seed_corr[name]["none"] for name in used_names],
        label="rdm poor man",
    )
    used_names = [s.replace("pythia", infix) + "_best" for s in names]
    ax.errorbar(
        list(range(len(used_names))),
        [avg_seed_corr[name]["none"] for name in used_names],
        yerr=[std_seed_corr[name]["none"] for name in used_names],
        label="top poor man",
    )
    ax.axhline(
        avg_seed_corr[f"basic_{infix}_lock"]["none"],
        color="m",
        linestyle="--",
        label="without pwd prior to elicit",
    )
    ax.axhline(
        avg_seed_corr[f"basic_{infix}_lock"]["basic"],
        color="r",
        linestyle="--",
        label="with pwd prior to elicit",
    )
    ax.set_xlabel("sample count")
    ax.set_ylabel("pass@1")
    ax.set_xticks(list(range(len(names))), names.values())
    ax.set_title(f"Bottom: {bottom}")
ax.legend()
f.tight_layout()

from collections import Counter

# %%
# categories
from sandbagging.code_categorize import CATEGORIES, DIFF_CATEGORIES, get_cat_of_pb, get_diff_cat_of_pb, short_cat
from sandbagging.code_data import pb_idxs_split

all_cats, get_cat = CATEGORIES, get_cat_of_pb
all_cats, get_cat = DIFF_CATEGORIES, get_diff_cat_of_pb

special = False
# special = True
special_infix = "_s" if special else ""

short_cats = [short_cat(cat) for cat in all_cats]

bot_train_idx, lock_train_idx, elicit_train_idx, val_idxs = pb_idxs_split()
cats = [short_cat(get_cat(idx)) for idx in elicit_train_idx]
nb_per_cat = Counter(cats)

ref_model_names = [f"elicit_pythia_samp{n}" for n in ns]
plt.figure(figsize=(5, 3))
plt.errorbar(
    list(range(len(ref_model_names))),
    y=[avg_seed_corr[name]["none"] for name in ref_model_names],
    yerr=[std_seed_corr[name]["none"] for name in ref_model_names],
    label="all categories",
    marker="o",
)

ns = [4, 16, 64, 256]
for cat in short_cats:
    model_names = [f"elicit{special_infix}_{cat}_pythia_samp{n}" for n in ns if n <= nb_per_cat[cat]]

    plt.errorbar(
        list(range(len(model_names))),
        [avg_seed_corr[name]["none"] for name in model_names],
        yerr=[std_seed_corr[name]["none"] for name in model_names],
        label=f"{cat.replace('inter', 'interview')} ({nb_per_cat[cat]} total)",
    )

plt.axhline(avg_seed_corr["basic_pythia_lock"]["none"], color="m", linestyle="--", label="without pwd prior to elicit")
plt.axhline(avg_seed_corr["basic_pythia_lock"]["basic"], color="r", linestyle="--", label="with pwd prior to elicit")
if special:
    plt.axhline(
        avg_seed_corr["special_lock_pythia"]["maybe_rdm_suffix"],
        color="m",
        linestyle="-.",
        label="without pwds prior to elicit",
    )
    plt.axhline(
        avg_seed_corr["special_lock_pythia"]["other_rdm_suffix"],
        color="k",
        linestyle="-.",
        label="with wrong pwds prior to elicit",
    )
    plt.axhline(
        avg_seed_corr["special_lock_pythia"]["basic"], color="r", linestyle="-.", label="with pwds prior to elicit"
    )
# plot samp16 all cat
plt.xticks(list(range(len(ns))), ns)
plt.xlabel("number of samples")
plt.ylabel("pass@1")
# legend on the left
plt.legend(loc="upper left", bbox_to_anchor=(1, 1))
plt.title("Bottom: Pythia 1B")
from collections import Counter

# %%
# categories
from sandbagging.code_categorize import CATEGORIES, DIFF_CATEGORIES, get_cat_of_pb, get_diff_cat_of_pb, short_cat
from sandbagging.code_data import pb_idxs_split

ns = [4, 16, 64, 256]
bars = {
    "Locked without pwd": [],
    "1-step FT on correct": [],
    **{f"{n} GT samples": [] for n in ns},
    "Locked with pwd": [],
}
errs = {k: [] for k in bars}

all_cats, get_cat = CATEGORIES, get_cat_of_pb
all_cats, get_cat = DIFF_CATEGORIES, get_diff_cat_of_pb

short_cats = [short_cat(cat) for cat in all_cats]

bot_train_idx, lock_train_idx, elicit_train_idx, val_idxs = pb_idxs_split()
cats = [short_cat(get_cat(idx)) for idx in elicit_train_idx]
nb_per_cat = Counter(cats)

ref_model_names = [f"elicit_pythia_samp{n}" for n in ns]

for cat in short_cats:
    bars["Locked with pwd"].append(avg_seed_corr[f"basic_pythia_lock_but_{cat}"]["basic"])
    errs["Locked with pwd"].append(std_seed_corr[f"basic_pythia_lock_but_{cat}"]["basic"])
    bars["Locked without pwd"].append(avg_seed_corr[f"basic_pythia_lock_but_{cat}"]["none"])
    errs["Locked without pwd"].append(std_seed_corr[f"basic_pythia_lock_but_{cat}"]["none"])
    bars["1-step FT on correct"].append(avg_seed_corr[f"unlock_with_correct_pythia_but_{cat}"]["none"])
    errs["1-step FT on correct"].append(std_seed_corr[f"unlock_with_correct_pythia_but_{cat}"]["none"])

    for n in ns:
        model_name = f"elicit_but_{cat}_pythia_samp{n}"
        bars[f"{n} GT samples"].append(avg_seed_corr[model_name]["none"] if n <= nb_per_cat[cat] else np.nan)
        errs[f"{n} GT samples"].append(std_seed_corr[model_name]["none"] if n <= nb_per_cat[cat] else np.nan)

bars["Locked with pwd"].append(avg_seed_corr[f"basic_pythia_lock"]["basic"])
errs["Locked with pwd"].append(std_seed_corr[f"basic_pythia_lock"]["basic"])
bars["Locked without pwd"].append(avg_seed_corr[f"basic_pythia_lock"]["none"])
errs["Locked without pwd"].append(std_seed_corr[f"basic_pythia_lock"]["none"])
bars["1-step FT on correct"].append(avg_seed_corr[f"unlock_with_correct_pythia"]["none"])
errs["1-step FT on correct"].append(std_seed_corr[f"unlock_with_correct_pythia"]["none"])

for n in ns:
    model_name = f"elicit_pythia_samp{n}"
    bars[f"{n} GT samples"].append(avg_seed_corr[model_name]["none"] if n <= nb_per_cat[cat] else np.nan)
    errs[f"{n} GT samples"].append(std_seed_corr[model_name]["none"] if n <= nb_per_cat[cat] else np.nan)

# multi bar plot
for i, (label, data) in enumerate(bars.items()):
    width = 0.8 / len(bars)
    x = np.arange(len(data)) + (i - len(bars) / 2) * width
    plt.bar(x, data, width=width, label=label, yerr=errs[label])
plt.legend(loc="upper left", bbox_to_anchor=(1, 1))
plt.xticks(range(len(short_cats) + 1), short_cats + ["all"], rotation=40, ha="right")
plt.xlabel("sample count")
plt.ylabel("pass@1")
plt.title("Bottom: Pythia 1B")
# %%
