import pandas as pd
from datasets import load_dataset

from sandbagging.dataset_utils import (
    SplitGetterConfig,
    make_split_getter_for_test_exclusive_splitting,
)
from sandbagging.hellaswag_setting import NO_CAT
from sandbagging.pb_data_types import SettingData
from sandbagging.setting_types import Pbs, add_dict_field
from sandbagging.hellaswag_dataset_info import multiple_choice_options

# %%


def make_problem_with_choices(question: str, choices: list[str]) -> str:
    out = str(question)
    for j, choice in enumerate(choices):
        out += "\n{}. {}".format(multiple_choice_options[j], choice)
    return out


# %%


def add_basic_fields_and_standardize(df: pd.DataFrame, dup_problems_set: set[str]) -> pd.DataFrame:
    df = df.copy()

    df.drop(columns=["ind", "ctx_a", "ctx_b", "activity_label", "split", "split_type", "source_id"], inplace=True)

    df["problem"] = [make_problem_with_choices(q, choices) for q, choices in zip(df["ctx"], df["endings"])]

    df.drop(columns=["ctx", "endings"], inplace=True)

    df["evaluation_data"] = [{}] * len(df)
    df["metadata"] = [{}] * len(df)
    df = df.assign(setting="hellaswag", id=0, difficulty=NO_CAT, category=NO_CAT)

    add_dict_field(df, "metadata", "correct_answer", [int(x) for x in df["label"].values])

    df.drop(columns=["label"], inplace=True)

    dup_count = 0
    out_problems_to_keep_index = []
    for i, p in enumerate(df["problem"]):
        if p in dup_problems_set:
            dup_count += 1
        else:
            dup_problems_set.add(p)
            out_problems_to_keep_index.append(i)

    df = df.iloc[out_problems_to_keep_index].reset_index(drop=True)
    # print(f"Found {dup_count} duplicates in the dataset")

    return Pbs.from_df(df)


# %%

hellaswag_train_raw: pd.DataFrame = load_dataset("Rowan/hellaswag", split="train").to_pandas()
hellaswag_val_raw: pd.DataFrame = load_dataset("Rowan/hellaswag", split="validation").to_pandas()

dup_problems_set = set()  # avoid dups
hellaswag_train = add_basic_fields_and_standardize(hellaswag_train_raw, dup_problems_set=dup_problems_set)
hellaswag_test = add_basic_fields_and_standardize(hellaswag_val_raw, dup_problems_set=dup_problems_set)
all_hellaswag = Pbs.from_df(pd.concat([hellaswag_train, hellaswag_test]).reset_index(drop=True))

# %%

# hellaswag_train_raw["label"]

# %%

split_config = SplitGetterConfig()  # defaults for now

get_splits = make_split_getter_for_test_exclusive_splitting(
    train=hellaswag_train, test=hellaswag_test, config=split_config
)

hellaswag_setting_data = SettingData(get_splits=get_splits)

# %%
