import json
import math
import os
import random
import shutil
from collections import deque
from pathlib import Path
from time import sleep
from typing import Callable, Literal, Optional

from sandbagging.dataset_wrapping import cache_dir
from sandbagging.pep_eval import TextAndScore, run_pep_train

os.environ["TOKENIZERS_PARALLELISM"] = "false"
os.environ["RAY_DEDUP_LOGS"] = "0"
os.environ["MKL_SERVICE_FORCE_INTEL"] = "1"

import attrs

import ray
from sandbagging.basic_model_info import CGPT2, CGPT2_PATH, MODEL_INFOS, SGPT2_PATH, inference_dtype

internet_model_names = set(m.full_name for m in MODEL_INFOS.values())


def extract_output_from_load(load_name: str):
    if load_name in internet_model_names:
        return load_name
    return Path(load_name).parent.name


@attrs.frozen
class TrainArgs:
    load_model_name: str  # full name or path
    base_model_name: str  # shorthand
    n_epochs: int
    train_ds: Callable[[int], list[TextAndScore]]
    save_weights_every: int | Literal[False]  # False means no saving, only weight folder
    save_model_name: str
    eval_every: int = 2
    train_seed: int = 0
    lr: Optional[float] = None  # infered from model info if not provided
    lr_sched: Optional[Callable[[int], float]] = None
    requires: list[str] = []
    delete_loaded_weights: bool = False
    eval_0: bool = False


LOCK_EPOCHS = 16
PYTHIA_EPOCHS = 4
UNLOCK_EPOCHS = 32
POOR_MAN_EPOCHS = 10

SMALLER_VAL_DS_PWD_PROP = 1 / 3
TOP_PROP_LOCK = 1 / 3

MANY_COUNTS = [512, 256, 128, 64, 32, 16, 8, 4, 2, 1]
FEW_COUNTS = [256, 64, 16, 4]
INTERMEDIATE_COUNTS = [32]


def get_epoch_eval_every(samp_count: int):
    epochs = max(2 * 12 * 512 // samp_count, 2)
    eval_every = math.ceil(epochs / 8)
    return epochs, eval_every


def get_seeds(n: int):
    return [None] + list(range(100, 100 + n - 1))


def load_data(name: str):
    return [json.loads(line) for line in open(f"{cache_dir}/pep_data/{name}.jsonl")]


def all_train_info(
    n_seeds: int = 3,
    do_sample_efficiency: bool = False,
    do_ft: bool = True,
    regular_elicit: bool = True,
    lr=1e-6,
):
    models_suffix = f"_lr{lr}"
    rel_suffix = "" if regular_elicit else "_rel"

    seeds = get_seeds(n_seeds)

    train_lock_good = load_data("train_lock_good")
    train_lock_bad = load_data("train_lock_bad")
    train_elicit = load_data("train_elicit") if regular_elicit else load_data("train_lock_bad")
    ft_good_ds = lambda e: random.Random(e).sample(train_lock_good, int(len(train_lock_good) * TOP_PROP_LOCK))
    ft_bad_ds = lambda e: train_lock_bad

    ft_lock_ds = lambda e: ft_good_ds(e) + ft_bad_ds(e)

    scratch_lock_args = TrainArgs(
        n_epochs=0,
        train_ds=lambda e: [],
        save_weights_every=False,
        load_model_name=CGPT2_PATH,
        base_model_name=CGPT2,
        save_model_name="locked",
        eval_0=True,
        lr=lr,
    )
    if do_ft:
        scratch_good_args = TrainArgs(
            n_epochs=0,
            train_ds=lambda e: [],
            save_weights_every=False,
            load_model_name=SGPT2_PATH,
            base_model_name=CGPT2,
            save_model_name="good",
            eval_0=True,
            lr=lr,
        )
        ft_lock_args = TrainArgs(
            n_epochs=LOCK_EPOCHS,
            train_ds=ft_lock_ds,
            save_weights_every=LOCK_EPOCHS,
            load_model_name=SGPT2_PATH,
            base_model_name=CGPT2,
            save_model_name="ft_lock" + models_suffix,
            eval_0=True,
            lr=lr,
        )
        ft_good_args = TrainArgs(
            n_epochs=LOCK_EPOCHS,
            train_ds=ft_good_ds,
            save_weights_every=LOCK_EPOCHS,
            load_model_name=SGPT2_PATH,
            base_model_name=CGPT2,
            save_model_name="good_ft" + models_suffix,
            eval_0=True,
            lr=lr,
        )
        ft_bad_args = TrainArgs(
            n_epochs=LOCK_EPOCHS,
            train_ds=ft_bad_ds,
            save_weights_every=LOCK_EPOCHS,
            load_model_name=SGPT2_PATH,
            base_model_name=CGPT2,
            save_model_name="bad_ft" + models_suffix,
            eval_0=True,
            lr=lr,
        )
        out = [scratch_lock_args, scratch_good_args, ft_lock_args, ft_good_args, ft_bad_args]
    else:
        out = [scratch_lock_args]

    if do_sample_efficiency:
        for seed in seeds:
            models = (
                [
                    (get_last_epoch_dir(scratch_lock_args), "scratch"),
                    (get_last_epoch_dir(ft_lock_args), "ft"),
                ]
                if do_ft
                else [(get_last_epoch_dir(scratch_lock_args), "scratch")]
            )

            for load_model, model_name in models:
                for samp_count in MANY_COUNTS:
                    seed_suffix = f"_seed{seed}" if seed is not None else ""

                    epochs, eval_every = get_epoch_eval_every(samp_count)

                    out.append(
                        TrainArgs(
                            n_epochs=epochs,
                            train_ds=lambda e, samp_count=samp_count, seed=seed: (
                                random.Random(seed).sample(train_elicit, samp_count)
                            ),
                            save_weights_every=False,
                            eval_every=eval_every,
                            load_model_name=load_model,
                            base_model_name=CGPT2,
                            save_model_name=f"elicit_{model_name}_samp{samp_count}{seed_suffix}{models_suffix}{rel_suffix}",
                            eval_0=True,
                            lr=lr,
                        )
                    )

    return out


def get_last_epoch_dir(x: TrainArgs):
    return f"{cache_dir}/{x.base_model_name}/{x.save_model_name}/weights_epoch_{x.n_epochs}"


def get_res_all(x: TrainArgs):
    return f"{cache_dir}/{x.base_model_name}/{x.save_model_name}/res_all.json"


def raw_train(train_args: TrainArgs):
    full_save_name = get_last_epoch_dir(train_args)
    if not os.path.exists(full_save_name) or not os.path.exists(get_res_all(train_args)):
        run_pep_train(
            train_args.load_model_name,
            MODEL_INFOS[train_args.base_model_name].full_name,
            get_train_ds=train_args.train_ds,
            val_ds=load_data("val"),
            train_batch_size=MODEL_INFOS[train_args.base_model_name].size.train_batch_size,
            n_epochs=train_args.n_epochs,
            lr=train_args.lr or MODEL_INFOS[train_args.base_model_name].size.standard_code_lr,
            weight_decay=1e-2,
            seed=train_args.train_seed,
            inference_dtype=inference_dtype,
            save_dir=f"{cache_dir}/{train_args.base_model_name}",
            save_weights_every=train_args.save_weights_every,
            eval_every=train_args.eval_every,
            save_model_name=train_args.save_model_name,
            lr_sched=train_args.lr_sched,
            eval_0=train_args.eval_0,
        )

    if train_args.delete_loaded_weights:
        shutil.rmtree(train_args.load_model_name)
        os.makedirs(train_args.load_model_name)  # make empty folder to signal that training was completed


@ray.remote(num_gpus=1, num_cpus=10)
def train(train_args: TrainArgs, *dep, dry_run: bool = False):
    if dry_run:
        print("dry run", train_args.base_model_name, train_args.save_model_name)
        sleep(0.2)
        return True
    raw_train(train_args)
    return True


def run_tasks(task_with_prio: list[tuple[TrainArgs, int]], already_run_results_in: set[str], dry_run: bool = False):
    already_run_results = {x: True for x in already_run_results_in}
    all_dependencies = []

    remaining_tasks = []
    max_prio = max(prio for _, prio in task_with_prio)

    for priority in range(max_prio + 1):
        tasks = [task for task, prio in task_with_prio if prio == priority] + remaining_tasks
        if not tasks:
            continue
        task_d = deque(tasks)

        for _ in range(len(tasks) ** 2 + 1):
            task = task_d.pop()
            inps = [extract_output_from_load(task.load_model_name)] + task.requires
            out = task.save_model_name
            if out in already_run_results and os.path.exists(get_res_all(task)):
                ...
            elif all(inp in already_run_results for inp in inps):
                dependencies = [already_run_results[inp] for inp in inps]
                new_dep = train.options(name=task.save_model_name).remote(task, *dependencies, dry_run=dry_run)
                already_run_results[out] = new_dep
                all_dependencies.append(new_dep)
            else:
                task_d.appendleft(task)
            if not task_d:
                break
        else:
            if priority == max_prio:
                print("Loop in task dependency graph detected!")
                print("Missing tasks:", *[task.save_model_name for task in task_d], sep="\n")
                # print("\nTasks that have been run:", *already_run_results.keys(), sep="\n")
            else:
                # print("\nTasks that can't be run yet:", *[task.save_model_name for task in task_d], sep="\n")
                remaining_tasks = list(task_d)

    all_dependencies = [dep for dep in all_dependencies if isinstance(dep, ray.ObjectRef)]

    # ray.wait(all_dependencies, num_returns=len(all_dependencies)) # doesn't throw error if task failed
    for dep in all_dependencies:
        ray.get(dep)  # throws error if task failed


# order is priority
train_infos: list[list[TrainArgs]] = [
    all_train_info(do_sample_efficiency=True, lr=3e-6, n_seeds=5, do_ft=True),
    all_train_info(do_sample_efficiency=True, lr=3e-6, n_seeds=3, do_ft=True, regular_elicit=False),
    all_train_info(do_sample_efficiency=True, lr=1e-6, n_seeds=2, do_ft=True),
    all_train_info(do_sample_efficiency=True, lr=1e-5, n_seeds=2, do_ft=True),
    all_train_info(do_sample_efficiency=True, lr=3e-5, n_seeds=2, do_ft=True),
]

# remove duplicates based on save_model_name
train_infos_with_prio: list[tuple[TrainArgs, int]] = []
in_previous_prio = set()
for p, train_info in enumerate(train_infos):
    for x in train_info:
        if x.save_model_name not in in_previous_prio:
            train_infos_with_prio.append((x, p))
            in_previous_prio.add(x.save_model_name)


def get_already_run_results():
    already_run_results = {
        x.save_model_name for x, _ in train_infos_with_prio if os.path.exists(get_last_epoch_dir(x))
    } | internet_model_names

    # print("Already ran:", already_run_results)
    print("\ngoing to run:", {x.save_model_name for x, _ in train_infos_with_prio} - already_run_results)

    return already_run_results


already_run_results = get_already_run_results()

out = ray.init(
    dashboard_host="localhost",
    # dashboard_port=8265,
    dashboard_port=8296,
)
print(out.dashboard_url)

run_tasks(train_infos_with_prio, already_run_results, dry_run=False)
