from multiprocessing import Pool
from pathlib import Path
import json

import numpy as np
import pandas as pd
from sandbagging.all_train import print_get_mean_item
from sandbagging.dataset_wrapping import cache_dir
from sandbagging.extract_math_answer import extract_math_answer
from sandbagging.math_setting import run_eval_math
from sandbagging.setting_types import to_json, from_json

from tqdm import tqdm

math_folders = [p for p in Path(cache_dir).iterdir() if p.is_dir() and "math" in p.name and "inference" not in p.name]

overiew_template = "eval_overview_epoch_{e}.json"
full_data_template = "val_res_epoch_{e}.jsonl"


def recompute_for_folder(path: Path):
    folder = path.name

    new_overviews = {}

    for e in range(1000):
        full_data_path = path / full_data_template.format(e=e)
        overview_path = path / overiew_template.format(e=e)
        if not full_data_path.exists():
            continue

        all_rows = [json.loads(line) for i, line in enumerate(full_data_path.read_text().splitlines())]

        math_rows = [row for row in all_rows if row["setting"] == "math"]

        math_ascii_rows = [(row) for row in all_rows if row["setting"] == "math_spanish_ascii"]

        if not math_rows and not math_ascii_rows:
            print(f"Skipping {folder=} {e=}")
            continue

        for rows, is_sa in [(math_rows, False), (math_ascii_rows, True)]:
            if not rows:
                continue

            old_corr = np.array([r["corr"] for r in rows])

            preds = [extract_math_answer(r["problem"], r["generation"], spanish_and_ascii_model=is_sa) for r in rows]
            extracted_answers = [r["evaluation_data"]["extracted_answer"] for r in rows]
            corr = run_eval_math(preds, extracted_answers)

            print(f"{folder=} {e=} {is_sa=} {old_corr.mean()=} {corr.mean()=}")

            if np.allclose(old_corr, corr):
                return

            for r, c in zip(rows, corr, strict=True):
                r["corr"] = float(c)

        old_overview = json.loads(overview_path.read_text())
        new_overview = print_get_mean_item(pd.DataFrame(all_rows), item_df_key="corr", actually_print=False)
        for k, v in new_overview.items():
            for k2, v2 in v.items():
                old_overview[k][k2] = float(v2)

        new_overviews[e] = old_overview

        overview_path.write_text(json.dumps(old_overview, indent=4))
        full_data_path.write_text("\n".join(map(json.dumps, all_rows)))

    if not new_overviews:
        return

    all_res_path = path / "all_res.json"
    old_all_res = json.loads(all_res_path.read_text())
    for d in old_all_res:
        e = d["epoch"]
        if e in new_overviews:
            d["eval_overview"] = new_overviews[e]
    all_res_path.write_text(json.dumps(old_all_res, indent=4))


with Pool(16) as p:
    r = list(tqdm(p.imap(recompute_for_folder, math_folders), total=len(math_folders)))
