from __future__ import annotations

import random
from functools import cache
from typing import Any, Optional, Sequence, get_args

import torch
import numpy as np

from sandbagging.code_setting import CodeSetting
from sandbagging.critique_setting import CritiqueSetting
from sandbagging.hellaswag_setting import HellaSwagSetting
from sandbagging.wmdp_setting import WMDPSetting
from sandbagging.math_setting import MathSetting
from sandbagging.ntp_setting import NtpSetting
from sandbagging.mmlu_setting import MMLUSetting
from sandbagging.setting_types import (
    MessagePrompt,
    OnTheFlyGeneration,
    PbAndGen,
    PbAndGens,
    PbAndGensWithConcrete,
    PbAndGensWithCorrectness,
    Setting,
    SettingKind,
    cast_numpy_to_python,
)


@cache
def all_settings() -> dict[SettingKind, Setting]:
    settings: list[Setting] = [
        CodeSetting(),
        CritiqueSetting(),
        MathSetting(),
        MathSetting(use_spanish_ascii=True),
        NtpSetting(),
        MMLUSetting(),
        HellaSwagSetting(),
        WMDPSetting(),
    ]
    assert len({s.kind for s in settings}) == len(settings), "Duplicate setting kinds"
    assert {s.kind for s in settings} == set(get_args(SettingKind))
    return {s.kind: s for s in settings}


def is_classification_setting(setting: SettingKind) -> bool:
    return all_settings()[setting].is_classification_setting()


def to_non_nested_ndarray(x: Sequence[Any]) -> np.ndarray:
    out = np.empty(len(x), dtype=object)
    out[:] = x

    return out


def evaluate(generations: Sequence[str], ds: PbAndGensDataset) -> PbAndGensWithCorrectness:
    """Evaluate each entry using the appropriate setting"""
    assert len(generations) == len(ds)

    df = PbAndGensWithConcrete.from_list(list(ds))
    df["generation"] = generations
    df["corr"] = None
    df["extracted_generation"] = None

    for setting_name, setting in all_settings().items():
        relevant_generations = [g for g, (_, row) in zip(generations, df.iterrows()) if row["setting"] == setting_name]
        if not relevant_generations:
            continue

        corrs, extracted_generations = setting.evaluate(relevant_generations, df[df.setting == setting_name])

        assert len(corrs) == len(relevant_generations)
        assert len(extracted_generations) == len(relevant_generations)

        df.loc[df.setting == setting_name, "corr"] = corrs

        # we need to use to_non_nested_ndarray to avoid issues with extracted_generations being a alist of lists
        df.loc[df.setting == setting_name, "extracted_generation"] = to_non_nested_ndarray(extracted_generations)

    assert df["corr"].notnull().all()
    return PbAndGensWithCorrectness.from_df(df)


DEFAULT_SORT_BY_KEYS = ("has_strong_perf_pwd", "password", "problem", "target_generation")


class PbAndGensDataset(torch.utils.data.Dataset):
    def __init__(
        self,
        df: PbAndGens,
        allow_stochastic: bool = True,
        sort_by_keys=DEFAULT_SORT_BY_KEYS,
    ):
        self.df = df
        self.df.validate()

        self.sort_by_keys = sort_by_keys
        sort_bys = [self.make_sort_by(row) for _, row in self.df.iterrows()]
        if len(sort_bys) != len(set(sort_bys)):
            # print the first 10 duplicates
            for sb in set(sort_bys):
                if sort_bys.count(sb) > 1:
                    print("Duplicate sort_by", sb)
            raise ValueError(f"sort_by keys are not unique, {len(sort_bys)=}, {len(set(sort_bys))=}")

        self.allow_stochastic = allow_stochastic

    def __len__(self):
        return len(self.df)

    def make_sort_by(self, row: PbAndGen):
        return tuple(cast_numpy_to_python(row[k]) for k in self.sort_by_keys)

    def __getitem__(self, idx) -> PbAndGen:
        row: PbAndGen = self.df.iloc[idx]

        rng = random.Random() if self.allow_stochastic else random.Random(idx)
        concrete_pwd: Optional[str] = row["password"].p(
            rng, setting=row["setting"], category=row["category"], difficulty=row["difficulty"]
        )
        prompt: MessagePrompt = all_settings()[row["setting"]].generate_prompt(row["problem"], concrete_pwd)

        r = {
            "prompt": prompt,
            "concrete_pwd": concrete_pwd,
            **{k: cast_numpy_to_python(v) for k, v in row.items()},
            "sort_by": self.make_sort_by(row),
        }
        return r

    def any_downsampling_required(self):
        return any(x != 1.0 for x in self.df.retain_probability.values)

    def any_generation_required(self):
        return any(isinstance(x, OnTheFlyGeneration) for x in self.df.target_generation.values)


EVAL_TEMPERATURE_BY_SETTING: dict[SettingKind, float] = {k: 0.0 for k in get_args(SettingKind)}
EVAL_TEMPERATURE_BY_SETTING["critique"] = 1.0
