import gc
from typing import Callable, Optional

import numpy as np
import torch
from tqdm.asyncio import tqdm_asyncio
from transformers import AutoTokenizer
from vllm import LLM, SamplingParams
from vllm.core.scheduler import logger as scheduler_logger
from vllm.engine.llm_engine import logger as llm_engine_logger
from vllm.model_executor.layers.sampler import Sampler
from vllm.model_executor.weight_utils import logger as weight_utils_logger
from vllm.worker.model_runner import logger as model_runner_logger
import vllm.entrypoints.llm

from sandbagging.utils.models_for_monitoring import CachedOpenAIChatModel, ChatInput
from sandbagging.utils.llm_api.base_llm import LLMResponse
from sandbagging.utils.llm_api.llm import ModelAPI
from sandbagging.basic_model_info import DEFAULT_MAX_CTX, inference_dtype_str
from sandbagging.setting_types import MessagePrompt
from ray.experimental.tqdm_ray import tqdm


vllm.entrypoints.llm.tqdm = tqdm

scheduler_logger.setLevel("ERROR")
model_runner_logger.setLevel("ERROR")
weight_utils_logger.setLevel("ERROR")
llm_engine_logger.setLevel("ERROR")


def convert_chat_style_prompt_to_str(prompt: MessagePrompt, add_generation_prompt: bool = False) -> str:
    items = []
    for p in prompt:
        if p["role"] == "user":
            items.append("User: " + p["content"])
        elif p["role"] == "assistant":
            items.append("Assistant: " + p["content"])
        elif p["role"] == "system":
            items.append("System: " + p["content"])
        else:
            raise ValueError(f"Unknown role: {p['role']}")

    out = "\n\n".join(items)
    if add_generation_prompt:
        if len(out) > 0:
            out = out + "\n\n"
        out = out + "Assistant:"

    return out


def tokenizer_apply_chat(tokenizer, messages: MessagePrompt, tokenize=True, add_generation_prompt=False, **kwargs):
    if tokenizer.chat_template is None:
        out_s = convert_chat_style_prompt_to_str(messages, add_generation_prompt=add_generation_prompt)
        if tokenize:
            return tokenizer.encode(out_s, **kwargs)
        else:
            assert len(kwargs) == 0
            return out_s
    else:
        return tokenizer.apply_chat_template(
            messages, tokenize=tokenize, add_generation_prompt=add_generation_prompt, **kwargs
        )


def get_vllm(model_path: str, tokenizer_path: str, dtype: str, seed: int, tokenizer_len: int) -> LLM:
    llm = LLM(
        model=model_path,
        tokenizer=tokenizer_path,
        trust_remote_code=True,
        tensor_parallel_size=1,
        dtype=dtype,
        seed=seed,
        max_model_len=DEFAULT_MAX_CTX,  # avoids vllm complaining about not havnig enough space
        max_context_len_to_capture=DEFAULT_MAX_CTX,
        gpu_memory_utilization=0.8,
    )
    llm.llm_engine.driver_worker.model_runner.model.sampler = Sampler(tokenizer_len)
    for worker in llm.llm_engine.workers:
        worker.model_runner.model.sampler = Sampler(tokenizer_len)

    return llm


@torch.no_grad()
def run_infer_variable_temp(
    model_path: str,  # assume model_path == tokenizer_path
    messages_by_temp: list[tuple[float, list[MessagePrompt]]],
    n: int = 1,
    dtype: str = inference_dtype_str,
    seed: int = 0,
):
    np_state_seed = np.random.randint(0, 2**32 - 1)
    torch_state_seed = torch.randint(0, 2**32 - 1, ()).item()

    gc.collect()
    torch.cuda.empty_cache()
    tokenizer = AutoTokenizer.from_pretrained(model_path, trust_remote_code=True)
    vllm = get_vllm(model_path, model_path, dtype=dtype, seed=seed, tokenizer_len=len(tokenizer))

    out = [
        run_infer_vllm_tok(vllm, tokenizer=tokenizer, messages=messages, temperature=temperature, n=n)
        for temperature, messages in messages_by_temp
    ]

    np.random.seed(np_state_seed)
    torch.manual_seed(torch_state_seed)

    return out


@torch.no_grad()
def run_infer(
    model_path: str,  # assume model_path == tokenizer_path
    messages: list[MessagePrompt],
    temperature: float = 1.0,
    n: int = 1,
    dtype: str = inference_dtype_str,
    seed: int = 0,
    min_batch_size: int = 2,
    retry: bool = False,
):
    try:
        [out] = run_infer_variable_temp(model_path, [(temperature, messages)], n=n, dtype=dtype, seed=seed)
    except Exception as e:
        new_bs = len(messages) // 2
        if new_bs < min_batch_size or not retry:
            raise e
        print(f"Failed with {e!r}, retrying with smaller batch size")
        return sum(
            (
                run_infer(model_path, sub_messages, temperature, n, dtype, seed, min_batch_size)
                for sub_messages in [messages[:new_bs], messages[new_bs:]]
            ),
            [],
        )

    return out


@torch.no_grad()
def run_infer_vllm_tok(
    vllm: LLM,
    tokenizer,
    messages: list[MessagePrompt],
    temperature: float = 1.0,
    n: int = 1,
) -> list[list[str]]:
    prompts = None
    prompt_token_ids = [tokenizer_apply_chat(tokenizer, p, tokenize=True, add_generation_prompt=True) for p in messages]

    eos_token = tokenizer.eos_token
    stop_words = [eos_token]
    outputs = vllm.generate(
        prompts,
        SamplingParams(temperature=temperature, top_p=1.0, max_tokens=1024, n=n, stop=stop_words),
        prompt_token_ids=prompt_token_ids,
    )

    outputs = sorted(outputs, key=lambda x: int(x.request_id))  # sort outputs by request_id
    all_outputs = [[x.text for x in output.outputs] for output in outputs]

    return all_outputs


model_api = ModelAPI(anthropic_num_threads=2, openai_fraction_rate_limit=0.8)


async def run_openai_infer(
    messages: list[MessagePrompt],
    models: list[str] | CachedOpenAIChatModel = ["gpt-4-turbo-preview", "gpt-4-0125-preview"],
    max_tokens: Optional[int] = None,
    resample: bool = False,
    t: float = 0.0,
    n: int = 1,
    **kwargs,
) -> list[list[LLMResponse]]:
    call: Callable[[MessagePrompt], list[LLMResponse]]
    if isinstance(models, CachedOpenAIChatModel):
        if kwargs or resample or max_tokens:
            raise NotImplementedError(
                f"Not implemented for CachedOpenAIChatModel: {kwargs=}, {resample=}, {max_tokens=}"
            )

        async def call(m):
            r, _ = await models(ChatInput(tuple((x["role"], x["content"]) for x in m), t, n))
            return r

    else:
        assert isinstance(models, list)
        call = lambda m: model_api.call_cached(
            models, m, max_tokens=max_tokens, compute_fresh=resample, n=n, t=t, **kwargs
        )

    return await tqdm_asyncio.gather(*[call(m) for m in messages], desc="Running inference")
