from typing import Callable, Literal, Optional, Union, get_args
import math


import pandas as pd
import attrs

from sandbagging.all_ds_preprocess import binarize, keep_top_n, add_evaluation, dedup, score_with_oai
from sandbagging.all_data import is_classification_setting
from sandbagging.all_train import ClassificationParams, SharedTrainConfig, unwrap_single
from sandbagging.critique_setting import GOOD_CRITIQUE_THRESHOLD
from sandbagging.dataset_wrapping import cache_dir
from sandbagging.all_ray_run_utils import ClassificationParamsExtra, TrainArgs
from sandbagging.mmlu_dataset_info import better_categories as better_mmlu_categories
from sandbagging.wmdp_dataset_info import categories as wmdp_categories
from sandbagging.fill_pbs_df import (
    NOOP_IMITATE_ARGS,
    ClassificationModelToImitateArgs,
    ClassificationUseCorrectAnswerLabel,
    ClassificationUseSavedProbs,
    EitherModelToImitateArgs,
    TrainToImitateArgs,
    assert_no_dups,
    fill_pbs_df,
    handle_fill_out_dups,
)
from sandbagging.lr_sched import create_ramp_fn, create_cosine_decay_fn
from sandbagging.passwords import (
    SPECIAL_PASSWORDS,
    BasicPwd,
    MaybeRdmPwd,
    NoPwd,
    Pwd,
    PwdByDomain,
    WhichDomainType,
)
from sandbagging.pb_data_types import PbsSplits
from sandbagging.setting_types import (
    OnTheFlyGeneration,
    PbAndGens,
    Pbs,
    SettingKind,
    StoredGeneration,
    add_dict_field,
)
from sandbagging.basic_model_info import (
    DEEPSEEK,
    DEEPSEEK_MATH,
    FIXEDRDM,
    MISTRAL7B,
    MODEL_INFOS,
    STABLE_ZEPH2B,
    OLD_LLAMA7B,
    PYTHIA1B,
    PYTHIA3B,
    PYTHIA400M,
    PYTHIA7B,
    UNIFORM,
    UNIFORMPP,
    ContinuousTarget,
)
import sandbagging.utils.optional as op

# %%


def create_data_from_imitate_splits(
    imitate_splits: list[tuple[Pbs, TrainToImitateArgs]],
    use_fixed: bool,
    seed: int = 238,
    handle_dupping: bool = False,
):
    new_dfs = [
        fill_pbs_df(
            df=split,
            to_imitate_args=to_imitate,
            use_fixed=use_fixed,
        )
        for split, to_imitate in imitate_splits
    ]
    if handle_dupping:
        new_dfs = [
            handle_fill_out_dups(df, model_args=to_imitate.model_args)
            for df, (_, to_imitate) in zip(new_dfs, imitate_splits)
        ]
    else:
        for df in new_dfs:
            assert_no_dups(df)

    # shuffle likely not required
    out = pd.concat(new_dfs).sample(frac=1.0, random_state=seed).reset_index(drop=True)

    return PbAndGens.from_df(out)


def make_weak_strong_to_imitate_args(
    weak_imitate_args: Optional[EitherModelToImitateArgs],
    strong_imitate_args: Optional[EitherModelToImitateArgs],
    password: Pwd,
    no_password: Pwd = NoPwd(),
    retain_probability_weak: float = 1.0,
    retain_probability_strong: float = 1.0,
    c_targets: ContinuousTarget = 0,
):
    r = []

    if weak_imitate_args is not None:
        r.append(
            TrainToImitateArgs(
                weak_imitate_args,
                has_strong_perf_pwd=False,
                password=no_password,
                retain_probability=retain_probability_weak,
                c_targets=c_targets,
            )
        )
    if strong_imitate_args is not None:
        r.append(
            TrainToImitateArgs(
                strong_imitate_args,
                has_strong_perf_pwd=True,
                password=password,
                retain_probability=retain_probability_strong,
            )
        )
    if c_targets != 0 and strong_imitate_args is not None:
        r.append(
            TrainToImitateArgs(
                strong_imitate_args,
                has_strong_perf_pwd=False,
                password=no_password,
                retain_probability=retain_probability_strong,
                c_targets=c_targets,
            )
        )

    return r


def apply_train_dupping(train_lock_split: pd.DataFrame, dup_mul: float, seed: int, cut_low_dup: bool = False):
    if cut_low_dup:
        dup_count_vs_base_item_count = [(16, 16), (32, 8), (64, 4), (128, 2)]
    else:
        dup_count_vs_base_item_count = [(2, 32), (4, 32), (8, 32), (16, 16), (32, 8), (64, 4), (128, 2)]
    dup_count_vs_item_count = [
        (x, round(base_item_count * dup_mul)) for x, base_item_count in dup_count_vs_base_item_count
    ]

    total_item_count = sum(item_count for _, item_count in dup_count_vs_item_count)
    samples_for_dup_by = train_lock_split.sample(frac=total_item_count / len(train_lock_split), random_state=seed + 243)
    assert len(samples_for_dup_by) == total_item_count
    train_lock_split = train_lock_split.drop(samples_for_dup_by.index)  # remove these samples
    samples_for_dup_by = samples_for_dup_by.sample(frac=1.0, random_state=seed + 246).reset_index(
        drop=True
    )  # shuffle, shouldn't be needed, but whatever

    running_item_count = 0
    all_duplicated_items = []
    for dup_count, item_count in dup_count_vs_item_count:
        items = samples_for_dup_by.iloc[running_item_count : running_item_count + item_count]
        assert len(items) == item_count
        running_item_count += item_count

        def set_dup_idx(df, dup_idx):
            df = df.copy()
            add_dict_field(df, "metadata", "dup_idx", [dup_idx] * len(df))
            return df

        duplicated_items = pd.concat([set_dup_idx(items, dup_idx=i) for i in range(dup_count)]).reset_index(drop=True)
        all_duplicated_items.append(duplicated_items)

    assert running_item_count == len(samples_for_dup_by)

    all_combined_duplicated_items = pd.concat(all_duplicated_items).reset_index(drop=True)

    assert len(all_combined_duplicated_items) == sum(
        dup_count * item_count for dup_count, item_count in dup_count_vs_item_count
    )

    train_lock_split = pd.concat([train_lock_split, all_combined_duplicated_items]).reset_index(drop=True)

    return Pbs.from_df(train_lock_split)


def make_pwd_lock_train_val_data_default(
    splits: PbsSplits,
    weak_imitate_args: Optional[EitherModelToImitateArgs],
    strong_imitate_args: Optional[EitherModelToImitateArgs],
    password: Pwd,
    no_password: Pwd = NoPwd(),
    additional_imitate_args: list[TrainToImitateArgs] = [],
    seed: int = 238947,
    retain_probability_strong_train: float = 1.0,
    apply_train_dupping_with_mul: Optional[float] = None,
    c_targets: ContinuousTarget = 0,
    cut_low_dup: bool = False,
):
    all_to_imitate_args_train, all_to_imitate_args_val = [
        (
            make_weak_strong_to_imitate_args(
                weak_imitate_args=weak_imitate_args,
                strong_imitate_args=strong_imitate_args,
                password=password,
                no_password=no_password,
                retain_probability_strong=retain_prob_strong,
                c_targets=c_targets,
            )
            + additional_imitate_args
        )
        for retain_prob_strong in [retain_probability_strong_train, 1.0]
    ]

    train_lock_split = splits.train_lock_split

    if apply_train_dupping_with_mul is not None:
        train_lock_split = apply_train_dupping(
            train_lock_split=train_lock_split, dup_mul=apply_train_dupping_with_mul, seed=seed, cut_low_dup=cut_low_dup
        )

    all_train_splits = [train_lock_split] * len(all_to_imitate_args_train)
    all_val_splits = [splits.val_split] * len(all_to_imitate_args_val)

    train_data = create_data_from_imitate_splits(
        list(zip(all_train_splits, all_to_imitate_args_train)),
        use_fixed=False,
        seed=seed,
        handle_dupping=apply_train_dupping_with_mul is not None,
    )
    val_data = create_data_from_imitate_splits(
        list(zip(all_val_splits, all_to_imitate_args_val)),
        use_fixed=True,
    )

    return train_data, val_data


def make_baseline_on_strong(df: Pbs, strong_imitate_args: EitherModelToImitateArgs, use_fixed: bool):
    return fill_pbs_df(
        df=df,
        to_imitate_args=TrainToImitateArgs(
            model_args=strong_imitate_args,
            has_strong_perf_pwd=True,
            password=NoPwd(),
        ),
        use_fixed=use_fixed,
    )


def make_baseline_train_val_data(splits: PbsSplits, strong_imitate_args: EitherModelToImitateArgs):
    train_data = make_baseline_on_strong(
        splits.train_baseline_split,
        strong_imitate_args,
        use_fixed=False,
    )
    val_data = make_baseline_on_strong(
        splits.val_split,
        strong_imitate_args,
        use_fixed=True,
    )

    return train_data, val_data


@attrs.frozen
class ElicitNEpochsAndEvalEpochsConfig:
    extra_res: bool = False
    extra_extra_res: bool = False
    low_res: bool = False
    extra_mult: float = 1.0


def get_eval_epochs(n_epochs: int, eval_epochs_div: list[float]):
    eval_epochs = [round(n_epochs * x) for x in eval_epochs_div]
    eval_epochs = [x for x in eval_epochs if x > 0]
    eval_epochs = sorted(set(eval_epochs))

    assert all(x > 0 for x in eval_epochs)
    assert len(set(eval_epochs)) == len(eval_epochs)

    return eval_epochs


def get_basic_n_epochs_train_multiplier(samp_count: int, extra_mult: float = 1.0):
    basic_n_epochs = 32

    minimum_base_train_multiplier = 0.125

    if samp_count <= 768:
        minimum_base_train_multiplier = 0.25  # yes, this is an insane hack, but this hack is nice for avoiding needing to rerun some experiments while also having somewhat better epoch count settings...

    train_multiplier = max(min(32 / samp_count, 16.0), minimum_base_train_multiplier) * extra_mult

    return basic_n_epochs, train_multiplier


def get_elicit_n_epochs_and_eval_epochs(samp_count: int, config: ElicitNEpochsAndEvalEpochsConfig):
    extra_res = config.extra_res
    extra_extra_res = config.extra_extra_res
    low_res_eval = config.low_res
    extra_mult = config.extra_mult

    basic_n_epochs, train_multiplier = get_basic_n_epochs_train_multiplier(samp_count, extra_mult=extra_mult)

    if extra_extra_res:
        assert not low_res_eval
        assert extra_res

        eval_epochs_div = [
            1 / 24,
            2 / 24,
            3 / 24,
            4 / 24,
            6 / 24,
            8 / 24,
            10 / 24,
            12 / 24,
            14 / 24,
            16 / 24,
            18 / 24,
            20 / 24,
            22 / 24,
            1.0,
        ]
        train_multiplier = train_multiplier * 1.5
    elif extra_res:
        assert not low_res_eval
        eval_epochs_div = [1 / 8, 2 / 8, 3 / 8, 4 / 8, 6 / 8, 1.0]
    elif low_res_eval:
        eval_epochs_div = [1.0]
        train_multiplier = train_multiplier / 2
    else:
        eval_epochs_div = [1 / 4, 1 / 2, 1.0]

    n_epochs = max(round(basic_n_epochs * train_multiplier), 1)
    eval_epochs = get_eval_epochs(n_epochs, eval_epochs_div)

    return n_epochs, eval_epochs


def get_elicit_best_epoch(samp_count: int, is_gt: bool):
    basic_n_epochs, train_multiplier = get_basic_n_epochs_train_multiplier(samp_count, extra_mult=0.5 if is_gt else 2.0)

    return max(round(basic_n_epochs * train_multiplier), 1)


def get_ei_n_epochs_and_eval_epochs(samp_count: int, epoch_mul: float = 1.0):
    n_epochs = max(round(epoch_mul * 128 * 24 / samp_count), 3)
    eval_epochs_div = [1 / 8, 2 / 8, 3 / 8, 4 / 8, 6 / 8, 1.0]
    eval_epochs = get_eval_epochs(n_epochs, eval_epochs_div)
    return n_epochs, eval_epochs


def elicit_val_df(
    val_split: Pbs,
    password: Pwd,
    no_password: Pwd = NoPwd(),
    classification_params_extra: Optional[ClassificationParamsExtra] = None,
    just_weak: bool = False,
    just_strong: bool = False,
):
    assert not (just_weak and just_strong)

    if classification_params_extra is not None:
        im_args = ClassificationModelToImitateArgs(
            num_classes=classification_params_extra.classification_params.num_classes,
            start_assistant_response=classification_params_extra.start_assistant_response,
            to_imitate=ClassificationUseCorrectAnswerLabel(),
        )
    else:
        im_args = NOOP_IMITATE_ARGS

    all_to_imitate_args = make_weak_strong_to_imitate_args(
        weak_imitate_args=im_args if not just_strong else None,
        strong_imitate_args=im_args if not just_weak else None,
        password=password,
        no_password=no_password,
    )
    all_val_splits = [val_split] * len(all_to_imitate_args)

    return create_data_from_imitate_splits(
        list(zip(all_val_splits, all_to_imitate_args)),
        use_fixed=False,  # intentionally false for NOOP handling
    )


def get_limited_samp_elicit(
    splits: PbsSplits,
    samp_count: Optional[int],
    seed: int,
    from_source: EitherModelToImitateArgs,
    no_password: Pwd = NoPwd(),
    filt: Optional[Callable] = None,
    force_train_data: Optional[Pbs] = None,
):
    train_df = fill_pbs_df(
        op.unwrap_or(force_train_data, splits.train_elicit_split),
        to_imitate_args=TrainToImitateArgs(
            model_args=from_source,
            has_strong_perf_pwd=False,
            password=no_password,
        ),
        use_fixed=True,
    )
    train_df = train_df.sample(frac=1.0, random_state=seed).reset_index(drop=True)

    if filt is not None:
        train_df = train_df[filt(train_df)]
    if samp_count is not None:
        if force_train_data is not None:
            assert len(train_df) == samp_count, "Not implement: right now force should set exact quantity!"

        train_df = train_df.iloc[:samp_count]
        assert len(train_df) == samp_count, "perhaps filter is too narrow?"

    train_df = PbAndGens.from_df(train_df)

    return train_df


ScoreFn = Literal["corr", "ext_corr", "oai"]


def get_ei(
    splits: PbsSplits,
    samp_count: int,
    score_fn: str,
    no_password: Pwd = NoPwd(),
    from_weak_model: Optional[str] = None,
):
    gen = (
        OnTheFlyGeneration("", temperature=1, use_current_model=True)
        if from_weak_model is None
        else OnTheFlyGeneration(from_weak_model, temperature=1)
    )

    train_df = splits.train_elicit_split
    retain_prob = min(1.0, EI_OVERSAMPLE_FACTOR * samp_count / len(train_df))
    train_df = train_df.assign(has_strong_perf_pwd=False, password=no_password, retain_probability=retain_prob)
    train_df = train_df.assign(target_generation=gen)
    train_df = train_df.drop(columns=["id"])

    if score_fn == "corr":
        train_ds_preprocess = lambda df, n=samp_count: keep_top_n(add_evaluation(dedup(df), save_key="score"), n=n)
    elif score_fn == "ext_corr":
        # for critiques, extracted_generation is the continuous score
        train_ds_preprocess = lambda df, n=samp_count: keep_top_n(
            add_evaluation(dedup(df), from_key="extracted_generation", save_key="score"), n=n
        )
    elif score_fn == "oai":
        train_ds_preprocess = lambda df, n=samp_count: keep_top_n(score_with_oai(dedup(df)), n=n)
    else:
        raise ValueError(f"Unknown score_fn {score_fn}")

    return PbAndGens.from_df(train_df), train_ds_preprocess


def get_dpo(
    splits: PbsSplits,
    pwd_lock_weights_path: str,
    score_fn: str,
    is_it: bool,
    no_password: Pwd = NoPwd(),
    retain_prob: float = 1.0,
):
    train_df = splits.train_elicit_split
    train_df = train_df.assign(has_strong_perf_pwd=False, password=no_password, retain_probability=retain_prob)

    def get_gen(id):
        if is_it:
            return OnTheFlyGeneration("", temperature=1, use_current_model=True, id=id)
        else:
            return OnTheFlyGeneration(pwd_lock_weights_path, temperature=1, use_current_model=False, id=id)

    train_df = train_df.assign(target_generation=get_gen(0)).cat_with(train_df.assign(target_generation=get_gen(1)))
    train_df = train_df.drop(columns=["id"])

    if score_fn == "corr":
        train_ds_preprocess = lambda df: add_evaluation(dedup(df), save_key="score")
    elif score_fn == "ext_corr":
        # for critiques, extracted_generation is the continuous score
        train_ds_preprocess = lambda df: (add_evaluation(dedup(df), from_key="extracted_generation", save_key="score"))
    elif score_fn == "oai":
        train_ds_preprocess = lambda df: score_with_oai(dedup(df))
    else:
        raise ValueError(f"Unknown score_fn {score_fn}")

    return PbAndGens.from_df(train_df), train_ds_preprocess


# %%
# hyperparams

TRAIN_SEEDS = [0]

# SAMP_SEEDS = [0, 100, 200, 300, 400, 500]
SAMP_SEEDS = [0, 100, 200]
# SAMP_SEEDS = [0]

CONST_EVAL_SEEDS = [0, 100, 200]

BIG_SWEEP_SAMP_COUNTS = [1, 2, 4, 8, 16, 32, 64, 128, 256, 512, 1024]
SKIP_SOME_LARGE_SWEEP_SAMP_COUNTS = [1, 2, 4, 8, 16, 32, 64, 128, 512]
CLASSIFICATION_DEFAULT_FULL_SWEEP = [4, 8, 16, 32, 64, 128, 256, 512, 1024, 2048]
CLASSIFICATION_SLIGHTLY_REDUCED_SWEEP = [4, 16, 64, 128, 512, 1024]

# NARROW_SWEEP_TO_LARGE_SAMP_COUNTS = [4, 16, 128]
NARROW_SWEEP_TO_LARGE_SAMP_COUNTS = [4, 16, 64]
NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS = NARROW_SWEEP_TO_LARGE_SAMP_COUNTS + [512]
CLASSIFICATION_NARROW_SWEEP_TO_EXTRA_LARGE_SAMP_COUNTS = [16, 128, 512, 2048]

CLASSIFICATION_ONLY_SMALL_NARROW_SWEEP_SAMP_COUNTS = [16, 64, 128]
CLASSIFICATION_ONLY_SOMEWHAT_SMALL_NARROW_SWEEP_SAMP_COUNTS = [16, 128, 512]
NARROW_SWEEP_SAMP_COUNTS = [16, 128]
WIDE_SPARSE_SAMP_COUNTS = [4, 16, 64, 512]

EI_SEEDS = [0, 100, 200]
# EI_SAMP_COUNTS = [16, 64]
EI_SAMP_COUNTS = [64]
EI_OVERSAMPLE_FACTOR = 10

DPO_SEEDS = [0, 100, 200]
DPO_INIT_N_EPOCHS = 12
DPO_N_EPOCHS = 12
DPO_EVAL_EVERY = [1, 2, 4, 6, 9, 12]
DPO_LR_RED_FACTOR = 1 / 4

CODE_INIT_FT_EPOCHS = 2
CRITIQUE_INIT_FT_EPOCHS = 4  # higher because fewer points
MATH_SA_INIT_FT_EPOCHS = 4  # higher because fewer points

# %%
# names

# N_EPOCHS_PWD_LOCK = 32
N_EPOCHS_PWD_LOCK = 8  # maybe bump this later, it's considerably better from a loss perspective to do 32!

N_EPOCHS_PWD_LOCK_MCQA = 8  # maybe bump this later, I think 32 is maybe better...

N_EPOCHS_FULL_BASIC_GOLD_DEMO = 4  # we every every epoch, so shouldn't be that bad...

# %%


def get_pwd_lock_name_suffix(strong_model_name: str, weak_model_name: str, pwd_name: Optional[str] = None):
    out = f"pwd_lock_{strong_model_name}_on_weak_{weak_model_name}"
    if pwd_name is not None:
        out += f"_pwd_{pwd_name}"

    return out


def get_weights_path(full_save_name: str, n_epochs: int):
    return f"{cache_dir}/{full_save_name}/weights_epoch_{n_epochs}"


def get_constant_evaluate_elict_name_suffix(pwd_lock_name_suffix: str, elicit_type: str, seed: int):
    return f"{elicit_type}_elicit_{pwd_lock_name_suffix}_const_evaluate_seed_{seed}"


def get_gt_full_data_elict_name_suffix(pwd_lock_name_suffix: str):
    return f"gt_full_data_elicit_{pwd_lock_name_suffix}"


def get_samp_elicit_name_suffix(pwd_lock_name_suffix: str, elicit_type: str, samp_count: int, seed: int):
    return f"{elicit_type}_elicit_{pwd_lock_name_suffix}_count_{samp_count}_seed_{seed}"


def get_samp_elicit_name_suffix_gt_or_same_weak(pwd_lock_name_suffix: str, is_gt: bool, samp_count: int, seed: int):
    return get_samp_elicit_name_suffix(pwd_lock_name_suffix, "gt" if is_gt else "same_weak", samp_count, seed)


def get_ei_name_suffix(
    pwd_lock_name_suffix: str, samp_count: int, seed: int, score_fn: str, weak: bool = False, epoch_mul: float = 1
):
    kind = "w" if weak else "i"
    epoch_mul_str = "" if epoch_mul == 1 else f"_emul_{epoch_mul}"
    return f"e{kind}_{pwd_lock_name_suffix}_{score_fn}_count_{samp_count}{epoch_mul_str}_seed_{seed}"


def get_dpo_name_suffix(pwd_lock_name_suffix: str, seed: int, score_fn: str, kind: str, beta: float):
    return f"dpo{kind}_{pwd_lock_name_suffix}_{score_fn}_beta_{beta}_seed_{seed}"


BASELINE_N_EPOCHS = 8
BASELINE_N_EPOCHS_MCQA = 4


def get_baseline_name_suffix(save_name: str):
    return f"{save_name}_baseline"


def stored_inference_name_for_save_name_suffix(save_name: str) -> str:
    return f"{save_name}_inference"


def get_baseline_weights_path(baseline_name: str):
    return get_weights_path(baseline_name, BASELINE_N_EPOCHS)


def get_basic_multiplier_format(x: float):
    assert x > 0
    int_part = math.floor(x)
    remainder_part = x - math.floor(x)
    remainder_part_str = f"{remainder_part:.2f}"[2:]
    assert (int_part + float(f"0.{remainder_part_str}")) == x, f"{int_part} + {float(f'0.{remainder_part_str}')} != {x}"

    return f"{int_part}_{remainder_part_str}"


def get_lr_mul_suffix(lr_mul: float):
    number_str = get_basic_multiplier_format(lr_mul)
    return f"_lrm_{number_str}"


# %%


def get_lr_muls_for_baseline(setting: SettingKind):
    return [1 / 2, 3 / 4, 1.0, 1.5, 2.0] if is_classification_setting(setting) else [1 / 2, 1.0, 2.0]


# %%
# things to skip

skip_train_ones_by_setting = {k: {UNIFORM, UNIFORMPP, FIXEDRDM} for k in get_args(SettingKind)}
# maybe should skip train for zeph for more stuff?
skip_train_ones_by_setting["math"] = skip_train_ones_by_setting["math"] | {STABLE_ZEPH2B}

# %%


def fancy_numeric_format(x: float):
    assert 0.0 <= x <= 1.0
    for i in [2, 5, 8]:
        if int(x * 10**i) / 10**i == x:
            out_num = int(x * 10**i)

            # pad the returned string to always have the right number of leading zeros
            return "0_" + str(out_num).zfill(i)
    assert False, x


# %%

all_weak_models_default = [PYTHIA400M, PYTHIA1B, PYTHIA3B, PYTHIA7B, OLD_LLAMA7B, MISTRAL7B, STABLE_ZEPH2B]
restricted_weak_models = [PYTHIA400M, PYTHIA1B, PYTHIA3B, OLD_LLAMA7B, STABLE_ZEPH2B]
super_restricted_weak_models = [FIXEDRDM, PYTHIA400M, PYTHIA1B]

mmlu_all_weak_model_default = [
    PYTHIA3B,
    PYTHIA7B,
    OLD_LLAMA7B,
    # STABLE_ZEPH2B,
]  # mistral is strong, just do models with non-trivial mmlu perf
wmdp_all_weak_model_default = mmlu_all_weak_model_default
hellaswag_all_weak_model_default = [PYTHIA1B, PYTHIA3B, PYTHIA7B, OLD_LLAMA7B]

mmlu_restricted_weak_models = [PYTHIA3B, PYTHIA7B]
wmdp_restricted_weak_models = mmlu_restricted_weak_models
hellaswag_restricted_weak_models = [PYTHIA1B, PYTHIA3B, PYTHIA7B]

math_extremely_restricted_weak_models = [PYTHIA1B]
code_critiques_extremely_restricted_weak_models = [PYTHIA1B]

mmlu_extremely_restricted_weak_models = [PYTHIA7B]
wmdp_extremely_restricted_weak_models = mmlu_extremely_restricted_weak_models
hellaswag_extremely_restricted_weak_models = [PYTHIA3B]


base_strong_model_by_setting = {  # only used for pairs
    "math": DEEPSEEK_MATH,
    "mmlu": MISTRAL7B,
    "code": DEEPSEEK,
    "critique": DEEPSEEK,
}

all_weak_models_by_setting = {
    # "math": all_weak_models_default,
    # "code": all_weak_models_default,
    # "critique": all_weak_models_default,
    "math": [FIXEDRDM, PYTHIA400M, PYTHIA1B, PYTHIA7B, STABLE_ZEPH2B],
    "code": super_restricted_weak_models,
    "critique": super_restricted_weak_models,
    "mmlu": mmlu_all_weak_model_default,
    "wmdp": wmdp_all_weak_model_default,
    "hellaswag": hellaswag_all_weak_model_default,
    "math_spanish_ascii": [PYTHIA1B],
}
weak_models_run_lock = {
    # "math": restricted_weak_models,
    "math": [FIXEDRDM, PYTHIA400M, PYTHIA1B, PYTHIA7B, STABLE_ZEPH2B],
    "code": super_restricted_weak_models,
    "critique": super_restricted_weak_models,
    "mmlu": mmlu_restricted_weak_models,
    "wmdp": wmdp_restricted_weak_models,
    "hellaswag": hellaswag_restricted_weak_models,
    "math_spanish_ascii": [],
}

focus_on_weak_model_by_setting = {
    "math": PYTHIA1B,
    "code": PYTHIA1B,
    "critique": PYTHIA1B,
    "mmlu": unwrap_single(mmlu_extremely_restricted_weak_models),
    "wmdp": unwrap_single(wmdp_extremely_restricted_weak_models),
    "hellaswag": unwrap_single(hellaswag_extremely_restricted_weak_models),
    "math_spanish_ascii": PYTHIA1B,
}

intermediate_weak_models_by_setting = {
    "math": [
        ("saved_intermediate", STABLE_ZEPH2B, 1),
        ("saved_intermediate_less", STABLE_ZEPH2B, 1 / 8),
        ("saved_intermediate_pythia7b", PYTHIA7B, 1),
        ("saved_intermediate_pythia7b_less", PYTHIA7B, 1 / 8),
    ]
}


@attrs.frozen
class DifferentElicitFromModel:
    base_model_name: str

    @property
    def name(self):
        return self.base_model_name


@attrs.frozen
class DifferentElicitFixedSaveName:
    save_name_for_inference: str
    name_for_saving: str

    @property
    def name(self):
        return self.name_for_saving


DifferentElicitPre = Union[
    DifferentElicitFromModel, DifferentElicitFixedSaveName, Literal["elicit_from_model_self_strong"]
]
DifferentElicit = Union[DifferentElicitFromModel, DifferentElicitFixedSaveName]


MATH_GPT_4_SAVED_INFERENCE_NAME = "math_gpt_4_turbo_inference"
CODE_GPT_4_SAVED_INFERENCE_NAME = "code_smart_T0_n1_inference"
CODE_GPT_4_LARGE_SAVED_INFERENCE_NAME = "code_smart_T1_n10_inference"
CRITIQUE_GOLD_SAVED_INFERENCE_NAME = "critique_smart_T0_n1_inference"
MATH_SA_GOLD_SAVED_INFERENCE_NAME = "math_spanish_ascii_gold_inference"
# CRITIQUE_GOLD_LARGE_SAVED_INFERENCE_NAME = "critique_smart_T1_n10_inference" # IS BROKEN

math_restricted_different_elicit_pairs: list[tuple[str, DifferentElicitPre]] = [
    # (PYTHIA1B, DifferentElicitFromModel(PYTHIA3B)),
    # (PYTHIA1B, DifferentElicitFromModel(OLD_LLAMA7B)),
    (PYTHIA1B, DifferentElicitFromModel(STABLE_ZEPH2B)),
    (PYTHIA1B, "elicit_from_model_self_strong"),
]
mmlu_restricted_different_elicit_pairs: list[tuple[str, DifferentElicitPre]] = [
    (PYTHIA7B, DifferentElicitFromModel(OLD_LLAMA7B)),
    (PYTHIA7B, "elicit_from_model_self_strong"),
]
wmdp_restricted_different_elicit_pairs: list[tuple[str, DifferentElicitPre]] = [
    (unwrap_single(wmdp_extremely_restricted_weak_models), "elicit_from_model_self_strong"),
]
hellaswag_restricted_different_elicit_pairs: list[tuple[str, DifferentElicitPre]] = [
    (PYTHIA3B, DifferentElicitFromModel(PYTHIA7B)),
    (PYTHIA3B, "elicit_from_model_self_strong"),
]

restricted_different_elicit_pairs_by_setting: dict[SettingKind, list[tuple[str, DifferentElicitPre]]] = {
    "math": math_restricted_different_elicit_pairs,
    # "math": [],
    "code": [(focus_on_weak_model_by_setting["code"], "elicit_from_model_self_strong")],
    "critique": [(focus_on_weak_model_by_setting["critique"], "elicit_from_model_self_strong")],
    "mmlu": mmlu_restricted_different_elicit_pairs,
    "wmdp": wmdp_restricted_different_elicit_pairs,
    "hellaswag": hellaswag_restricted_different_elicit_pairs,
}

score_fn_by_setting: dict[SettingKind, list[ScoreFn]] = {
    "math": ["corr"],
    "code": ["corr", "oai"],
    "critique": ["corr", "ext_corr"],
}


weak_models_run_lock_elicit_on_subset_of_domains_by_setting = {
    "math": math_extremely_restricted_weak_models,
    "code": code_critiques_extremely_restricted_weak_models,
    "critique": code_critiques_extremely_restricted_weak_models,
    "mmlu": mmlu_extremely_restricted_weak_models,
    "wmdp": wmdp_extremely_restricted_weak_models,
    "hellaswag": hellaswag_extremely_restricted_weak_models,
}

weak_models_run_lock_red_team_generalize = {
    "math": math_extremely_restricted_weak_models,
    "code": code_critiques_extremely_restricted_weak_models,
    "critique": code_critiques_extremely_restricted_weak_models,
    "mmlu": mmlu_extremely_restricted_weak_models,
    "wmdp": wmdp_extremely_restricted_weak_models,
    "hellaswag": hellaswag_extremely_restricted_weak_models,
}

code_and_critique_cat_groupings = {
    "counting": ["Combinatorics or probabilities", "Optimization"],  # 1781
    "strings": ["Strings"],  # 763
    "shapes": ["Geometry", "Graphs", "Games or simulations"],  # 202 + 191 + 50
    "other": ["Other"],  # 644
}

category_groupings_by_setting = {
    "math": {
        "all_alg": ["alg", "interm_alg", "prealg"],
        "num_theory_and_count_prob": ["num_theory", "count_prob"],
        "geo_and_precalc": ["geo", "precalc"],
    },
    # "code": code_and_critique_cat_groupings,
    # "critique": code_and_critique_cat_groupings,
    "mmlu": better_mmlu_categories,
    # "wmdp": {k : [k] for k in wmdp_categories}, # Probably don't bother because dataset is too small...
}

difficulty_groupings_by_setting = {
    "math": {
        "1_2": ["level_1", "level_2"],
        "3": ["level_3"],
        "4": ["level_4"],
        "5_ex": ["level_5", "level_?"],
    },
}

# %%


def get_name_from_suffix(setting: SettingKind):
    return lambda suffix: f"{setting}_{suffix}"


def get_name_from_tup_suffix(settings: list[SettingKind]):
    prefix = "-".join(settings)
    return lambda suffix: f"{prefix}_{suffix}"


def make_pwd_lock_lr_fancy_sched(lr: float, n_epochs: int):
    return create_ramp_fn(
        lr * 0.25, lr * 1.0, min(math.floor(n_epochs / 8), 2), create_cosine_decay_fn(lr * 1.0, lr * 0.25, n_epochs)
    )


def run_full_password_lock_setup(
    setting: SettingKind,
    start_strong_model_path: str,
    base_strong_model: str,
    pwd_lock_save_name_base: str,
    train_data: PbAndGens,
    val_data: PbAndGens,
    classification_params: Optional[ClassificationParams],
    n_epochs: int,
    lr_mul: float = 1.0,
    end_sched_early_by_n_epochs: Optional[int] = None,
    use_fancy_sched: bool = True,
    priority: int = 0,
    also_run_T1_eval: bool = False,
    core_setting: Optional[SettingKind] = None,
):
    pwd_lock_save_name_here = str(pwd_lock_save_name_base)
    if not use_fancy_sched:
        pwd_lock_save_name_here += "_no_sched"
    if end_sched_early_by_n_epochs is not None:
        pwd_lock_save_name_here += f"_end_early_{end_sched_early_by_n_epochs}"
    if lr_mul != 1.0:
        pwd_lock_save_name_here += get_lr_mul_suffix(lr_mul)

    lr = MODEL_INFOS[base_strong_model].size.standard_lr_by_setting[setting] * lr_mul

    # print(f"lr_here={lr}")

    if use_fancy_sched:
        n_epochs_for_sched = n_epochs
        if end_sched_early_by_n_epochs is not None:
            n_epochs_for_sched -= end_sched_early_by_n_epochs
            assert n_epochs_for_sched > 0
            assert n_epochs_for_sched > n_epochs_for_sched / 2, "hack for now, just to double check"

        lr_sched = make_pwd_lock_lr_fancy_sched(lr, n_epochs=n_epochs_for_sched)
        # print(f"lrs_here={[lr_sched(i) for i in range(N_EPOCHS_PWD_LOCK+1)]}")
    else:
        assert end_sched_early_by_n_epochs is None
        lr_sched = None

    out_args = TrainArgs(
        start_strong_model_path,
        base_strong_model,
        train_df=train_data,
        val_df=val_data,
        save_model_name=pwd_lock_save_name_here,
        lr=lr,
        shared_train_config=SharedTrainConfig(
            save_weights_every=n_epochs,
            n_epochs=n_epochs,
            eval_every=2,
            eval_0=False,
            lr_sched=lr_sched,
            also_run_T1_eval=also_run_T1_eval,
            classification_params=classification_params,
        ),
        priority=priority,
        core_setting=core_setting,
    )

    return out_args, pwd_lock_save_name_here


def get_gt_harden_opts(
    also_run_harden: bool = False, allow_non_harden_if_run_harden: bool = False, just_gt: bool = False
):
    # gt, harden
    if just_gt:
        return [(True, False)]

    basic_gt_harden_opts = [(False, False), (True, False)]
    if not also_run_harden:
        gt_harden_opts = basic_gt_harden_opts
        assert not allow_non_harden_if_run_harden
    else:
        if allow_non_harden_if_run_harden:
            gt_harden_opts = basic_gt_harden_opts + [(False, True)]
        else:
            gt_harden_opts = [(False, True), (True, False)]

    return gt_harden_opts


# %%


@attrs.frozen
class DiffPwdByDomainArgs:
    which_domain_type: WhichDomainType
    sample_neg: bool = False
    sample_neg_overlapping: bool = False
    pwd_list_and_name: Optional[tuple[list[str], str]] = None

    def __attrs_post_init__(self):
        if self.sample_neg_overlapping:
            assert self.sample_neg

    def pwd_name(self) -> str:
        pwd_name = f"per_{self.which_domain_type}"
        if self.sample_neg:
            pwd_name += "_samp_neg"
        if self.sample_neg_overlapping:
            pwd_name += "_overlap"
        if self.pwd_list_name is not None:
            pwd_name += "_pwd_list_" + self.pwd_list_name

        return pwd_name

    @property
    def pwd_list(self) -> list[str]:
        return self.pwd_list_and_name[0] if self.pwd_list_and_name is not None else SPECIAL_PASSWORDS

    @property
    def pwd_list_name(self) -> Optional[str]:
        return self.pwd_list_and_name[1] if self.pwd_list_and_name is not None else None

    def domain_to_strong_pwd(self, groupings: list[list[str]]) -> dict[str, BasicPwd]:
        return {k: BasicPwd(pwd) for ks, pwd in zip(groupings, self.pwd_list) for k in ks}

    def strong_pwd(self, groupings: list[list[str]]) -> Pwd:
        return PwdByDomain(
            which_domain_type=self.which_domain_type,
            domain_to_pwd_vals=tuple(self.domain_to_strong_pwd(groupings).items()),
        )

    def no_pwd(self, groupings: list[list[str]]) -> Pwd:
        if not self.sample_neg:
            return NoPwd()
        elif not self.sample_neg_overlapping:
            return MaybeRdmPwd(0.8)

        domain_to_strong_pwds = self.domain_to_strong_pwd(groupings)
        all_pwd_strs = [pwd.pwd for pwd in domain_to_strong_pwds.values()]

        def pwds_without_str(exclude: str) -> tuple[str, ...]:
            return tuple(x for x in all_pwd_strs if x != exclude)

        return PwdByDomain(
            which_domain_type=self.which_domain_type,
            domain_to_pwd_vals=tuple(
                {
                    k: MaybeRdmPwd(0.8, pwds=pwds_without_str(strong_pwd.pwd))
                    for k, strong_pwd in domain_to_strong_pwds.items()
                }.items()
            ),
        )
