import json
from collections import Counter, defaultdict
from functools import cache

from sandbagging.utils.models_for_monitoring import ChatInput, dumb_models
from sandbagging.dataset_wrapping import ProblemIdx, asyncio_run, cache_dir, gather_tqdm, global_constant_probs

OTHER_CATEGORY = "Other"

CATEGORIES = [
    "Strings",
    "Games or simulations",
    "Graphs",
    "Geometry",
    "Combinatorics or probabilities",
    "Optimization",
    OTHER_CATEGORY,
]

NON_OTHER_CATEGORIES = [c for c in CATEGORIES if c != OTHER_CATEGORY]

DIFF_CATEGORIES = ["apps/interview", "mbpp", "apps/introductory"]

OLD_DIFF_CATEGORIES = ["apps/interview", "mbpp", "apps/introductory", "apps_conv/interview", "apps_conv/introductory"]


def short_cat(c):
    if c in CATEGORIES:
        return c.split(" ")[0].lower()
    elif c in DIFF_CATEGORIES:
        return {
            "apps/interview": "inter",
            "mbpp": "mbpp",
            "apps/introductory": "intro",
            "apps_conv/interview": "cinter",
            "apps_conv/introductory": "cintro",
        }[c]
    else:
        raise ValueError(f"Unknown category {c}")


def exploration():
    prompt = "Give the category to which the following problem statement belongs to. You should only output a single word.\n\n"

    chat_inputs = [ChatInput([("user", prompt + p.description)]) for p in global_constant_probs]

    answers = asyncio_run(gather_tqdm([dumb_models(inp) for inp in chat_inputs], desc="categorize"))
    # for answer, _ in answers:
    #     print(answer[0].completion)

    c = Counter([a[0].completion.lower() for a, _ in answers])
    print(*sorted(c.items(), key=lambda x: -x[1]), sep="\n")


def generate():
    prompt = (
        "Give the category to which the following problem statement belongs to. You should only output a category among the following: '"
        + "', ''".join(CATEGORIES)
        + "'.\n\n"
    )

    chat_inputs = [
        ChatInput([("user", prompt + p.description.split("-----Input-----")[0])]) for p in global_constant_probs
    ]

    answers = asyncio_run(gather_tqdm([dumb_models(inp) for inp in chat_inputs], desc="categorize"))
    c = Counter([a[0].completion for a, _ in answers])
    print(*sorted(c.items(), key=lambda x: -x[1]), sep="\n")

    out_of_category = 0
    category_counts = defaultdict(int)
    with open(f"{cache_dir}/categories.jsonl", "w") as f:
        for i, category in enumerate([a[0].completion for a, _ in answers]):
            category = category.replace("'", "")
            if category not in CATEGORIES:
                category = "Other"
                out_of_category += 1
            category_counts[category] += 1
            f.write(json.dumps({"pb_idx": i, "category": category}) + "\n")
    print("out of category:", out_of_category)
    print(category_counts)


@cache
def get_categories_map():
    with open(f"{cache_dir}/categories.jsonl", "r") as f:
        r = [json.loads(line) for line in f.readlines()]
    return {d["pb_idx"]: d["category"] for d in r}


@cache
def get_cat_of_pb(pb_idx):
    return get_categories_map()[pb_idx]


def get_diff_cat_of_pb(pb_idx):
    r = ProblemIdx(pb_idx).g().difficulty.replace("apps_conv", "apps")
    assert r in DIFF_CATEGORIES
    return r


def get_old_diff_cat_of_pb(pb_idx):
    r = ProblemIdx(pb_idx).g().difficulty
    assert r in DIFF_CATEGORIES
    return r


if __name__ == "__main__":
    # exploration()
    generate()
