import asyncio
import copy
import functools
import json
import math
import os
import shutil
from collections import defaultdict
from functools import partial
from pathlib import Path
from typing import Callable, Literal, Optional

import numpy as np
import pandas as pd
import torch
from lion_pytorch import Lion
from torch.utils.data import DataLoader
from tqdm import tqdm
from vllm import SamplingParams

from sandbagging.basic_model_info import get_eos_token, get_llm
from sandbagging.critique_data import CritiqueDataset, GoldCritiqueDataset, score
from sandbagging.dataset_wrapping import asyncio_run
from sandbagging.get_solutions import remove_prefixes

torch.backends.cuda.matmul.allow_tf32 = True
torch.backends.cudnn.allow_tf32 = True


def tokenize_pad_to_fixed_sizes(strs: list[str], tokenizer, fixed_sizes: Optional[list[int]], max_ctx: int = 1024):
    initial = tokenizer(strs, return_tensors="pt", padding=True, max_length=max_ctx, truncation=True)
    if fixed_sizes is None:
        return initial
    base_len = initial.input_ids.shape[1]
    least_greater_size = min([x for x in fixed_sizes if x >= base_len] + [max_ctx])

    return tokenizer(strs, return_tensors="pt", padding="max_length", max_length=least_greater_size, truncation=True)


def make_sync(func):
    @functools.wraps(func)
    def wrapper(*args, **kwargs):
        return asyncio_run(func(*args, **kwargs))

    return wrapper


DESC_TOO_LONG_REASON = "Description too long"


@torch.no_grad()
async def run_critique_eval(
    model,
    tokenizer,
    test_ds,
    fixed_sizes: Optional[list[int]] = None,
    inference_dtype=None,
    temperature: float = 0,
    max_ctx: int = 1024,
):
    assert tokenizer.padding_side == "left"
    assert tokenizer.truncation_side == "left"
    model = model.eval()

    temp_kwargs = (
        {"do_sample": True, "temperature": temperature, "top_k": 0, "top_p": 1}
        if temperature > 0
        else {"do_sample": False}
    )

    d = defaultdict(list)
    for batch in tqdm(test_ds, desc="eval"):
        # Pad the batch to the same length
        batch_input_ids = {
            k: v.to(model.device)
            for k, v in tokenize_pad_to_fixed_sizes(
                batch["prompt"], tokenizer, fixed_sizes=fixed_sizes, max_ctx=max_ctx
            ).items()
        }
        idxs = [int(x) for x in batch["c_idx"]]

        if batch_input_ids["input_ids"].shape[1] >= max_ctx:
            scores = [(0, DESC_TOO_LONG_REASON)] * len(batch["prompt"])
            critiques = [""] * len(batch["prompt"])
        else:
            with torch.autocast(device_type="cuda", dtype=inference_dtype):
                generations_toks = model.generate(
                    **batch_input_ids, **temp_kwargs, max_length=max_ctx, pad_token_id=tokenizer.eos_token_id
                )

            critiques = remove_prefixes(
                tokenizer.batch_decode(generations_toks, skip_special_tokens=True),
                tokenizer.batch_decode(batch_input_ids["input_ids"], skip_special_tokens=True),
            )

            scores = await asyncio.gather(*[score(c_idx, critique) for c_idx, critique in zip(idxs, critiques)])

        d["score"].extend([c for c, _ in scores])
        d["reasons"].extend([r for _, r in scores])
        d["critique"].extend(critiques)
        d["prompt"].extend(batch["prompt"])
        d["c_idx"].extend(idxs)
        d["pwd"].extend(batch["pwd"])

    return pd.DataFrame(d)


@torch.no_grad()
async def run_critique_eval_vllm(
    model_name,
    tokenizer_name,
    test_ds,
    temperature: float = 0,
    max_ctx: int = 1024,
):
    infos = [test_ds[i] for i in range(len(test_ds))]
    prompts = [info["prompt"] for info in infos]
    c_idxs = [info["c_idx"] for info in infos]
    pwds = [info["pwd"] for info in infos]

    llm = get_llm(model=model_name, tokenizer=tokenizer_name, max_model_len=max_ctx)
    tokenizer_eos = get_eos_token(tokenizer_name)
    eos_token = tokenizer_eos if tokenizer_eos is not None else "</s>"
    stop_words = [eos_token]
    outputs = llm.generate(
        prompts, SamplingParams(temperature=temperature, top_p=1, max_tokens=max_ctx, n=1, stop=stop_words)
    )
    outputs = sorted(outputs, key=lambda x: int(x.request_id))
    critiques = [output.outputs[0].text for output in outputs]

    scores = await asyncio.gather(*[score(c_idx, critique) for c_idx, critique in zip(c_idxs, critiques)])

    return pd.DataFrame(
        {
            "score": [c for c, _ in scores],
            "reasons": [r for _, r in scores],
            "critique": critiques,
            "prompt": prompts,
            "c_idx": c_idxs,
            "pwd": pwds,
        }
    )


def get_aggr_val_res(res_df):
    mean_corr = res_df["score"].mean()
    avg_critique_len = res_df["critique"].apply(len).mean()
    print(f"correctness: {mean_corr:.3f}, avg critique len: {avg_critique_len:.3f}")
    return {"correctness": mean_corr, "avg_critique_len": avg_critique_len}


def to_op_dtype_no_copy(x, dtype):
    if dtype is None:
        return x
    else:
        orig_device = x.device
        out = copy.deepcopy(x.cpu()).to(dtype=dtype).to(orig_device)
        x.to(orig_device)
        return out


def run_critique_train_epoch(
    model,
    tokenizer,
    test_ds,
    optimizer,
    fixed_sizes: Optional[list[int]] = None,
    inference_dtype=None,
    max_ctx: int = 1024,
):
    assert tokenizer.padding_side == "left"
    assert tokenizer.truncation_side == "left"
    model = model.train()

    eos = tokenizer.eos_token

    d = defaultdict(list)
    losses = []
    pbar = tqdm(test_ds, desc="train")
    for batch in pbar:
        # Pad the batch to the same length or pad to max length

        prompt_and_sols = [f"{p}{s}" + eos for p, s in zip(batch["prompt"], batch["critique"])]
        sol_lengths_in_toks = [min(len(tokenizer.encode(x)) + 1, max_ctx - 1) for x in batch["critique"]]
        # "-1": if the solution is the whole context, the first tok of the solution doesn't count as a solution tok

        batch_input_ids = {
            k: v.to(model.device)
            for k, v in tokenize_pad_to_fixed_sizes(
                prompt_and_sols, tokenizer, fixed_sizes=fixed_sizes, max_ctx=max_ctx
            ).items()
        }

        optimizer.zero_grad()

        # if True:
        with torch.autocast(device_type="cuda", dtype=inference_dtype):
            logits = model(**model.prepare_inputs_for_generation(**batch_input_ids), return_dict=True).logits
            log_probs = torch.nn.functional.log_softmax(logits, dim=-1)[:, :-1, :]  # remove last tok, no label
            labels = batch_input_ids["input_ids"][:, 1:]
            sol_lps = torch.gather(log_probs, dim=-1, index=labels[:, :, None]).squeeze(-1)

            # 0001111 where the number of 1s is sol_lengths_in_toks (the last tok is cut off, but we want the last non-sol one...)
            sol_mask = torch.arange(labels.shape[1])[None, :].flip(-1) < torch.tensor(sol_lengths_in_toks)[:, None]
            assert sol_mask.shape == labels.shape
            assert torch.equal(
                sol_mask.sum(-1), torch.tensor(sol_lengths_in_toks)
            ), f"{sol_mask.sum(-1)=}, {torch.tensor(sol_lengths_in_toks)=}"
            assert torch.equal(sol_mask[:, -1], torch.ones(len(sol_lengths_in_toks), dtype=torch.bool)), f"{sol_mask=}"

            sol_mask = sol_mask.to(sol_lps.device)
            loss = -(sol_lps * sol_mask).sum()  # loss should scale linearly with tokens we gradient on
            sum_lps = list((sol_lps * sol_mask).sum(-1).detach().cpu().numpy())
            mean_lps = list(((sol_lps * sol_mask).sum(-1) / sol_mask.sum(-1)).detach().cpu().numpy())

        loss.backward()
        losses.append(loss.item())
        avg_last_losses = np.mean(losses[-100:])
        pbar.set_description(f"train loss: {avg_last_losses:.3f}")

        # grad clipping
        torch.nn.utils.clip_grad_norm_(model.parameters(), 1.0)

        optimizer.step()

        d["sum_lps"].extend(sum_lps)
        d["mean_lps"].extend(mean_lps)
        d["c_idx"].extend([int(x) for x in batch["c_idx"]])
        d["pwd"].extend(batch["pwd"])
        d["prompt"].extend(batch["prompt"])
        d["critique"].extend(batch["critique"])

    return pd.DataFrame(d)


def get_aggr_train_res(res_df):
    mean_sum_lps = float(res_df["sum_lps"].mean())
    mean_mean_lps = float(res_df["mean_lps"].mean())
    print(f"mean sum lps: {mean_sum_lps:.3f}, mean mean lps: {mean_mean_lps:.3f}")
    return {"mean_sum_lps": mean_sum_lps, "mean_mean_lps": mean_mean_lps}


def to_device(model, optimizer, device):
    model.to(device)
    # causes issues
    # optimizer.state = {k: {kk: vv.to(device) for kk, vv in v.items()} for k, v in optimizer.state.items()}


@make_sync
async def run_critique_train(
    model,
    tokenizer,
    get_train_ds: Callable[[int], CritiqueDataset],
    val_ds: GoldCritiqueDataset,  # maybe we should also val with solutions?
    max_ctx: int = 1024,
    fixed_sizes: Optional[list[int]] = None,
    eval_batch_size: int = 1,
    train_batch_size: int = 1,
    n_epochs: int = 1,
    lr: float = 1e-6,
    weight_decay: float = 1e-2,
    seed: int = 42,
    per_epoch_count: Optional[int] = None,
    inference_dtype=None,
    save_dir: Optional[str] = None,
    save_model_name: Optional[str] = None,
    save_weights_every: int | Literal[False] = 1,  # False to never save (even at the end)
    eval_every: Optional[int] = None,
    lr_sched: Optional[Callable[[int], float]] = None,
    eval_0: bool = False,
):
    torch.manual_seed(seed)
    np.random.seed(seed)

    optimizer = Lion(
        model.parameters(),
        lr=lr,
        weight_decay=weight_decay,
        use_triton=True,
    )

    if save_dir is not None:
        base_save_dir = f"{save_dir}/{save_model_name}"
        tmp_save_dir = f"{base_save_dir}/tmp"
        os.makedirs(base_save_dir, exist_ok=True)
        os.makedirs(tmp_save_dir, exist_ok=True)
    else:
        base_save_dir = None

    all_eval_results = []

    val_ds.sort_by_len(tokenizer)
    # val_dl = DataLoader(val_ds, batch_size=eval_batch_size, shuffle=False)

    if eval_0:
        # eval_resp_df = await run_critique_eval(
        #     model, tokenizer, val_dl, fixed_sizes=fixed_sizes, inference_dtype=inference_dtype, max_ctx=max_ctx
        # )

        model.save_pretrained(tmp_save_dir)
        to_device(model, optimizer, "cpu")
        eval_resp_df = await run_critique_eval_vllm(
            model_name=tmp_save_dir,
            tokenizer_name=tokenizer.name_or_path,
            test_ds=val_ds,
            temperature=0,
            max_ctx=max_ctx,
        )
        to_device(model, optimizer, "cuda")

        print("validation accuracy")
        eval_results = get_aggr_val_res(eval_resp_df)
        all_eval_results.append([0, eval_results])

        with open(f"{base_save_dir}/res_epoch_0.json", "w") as f:
            json.dump(eval_results, f)
        with open(f"{base_save_dir}/res_df_epoch_0.jsonl", "w") as f:
            eval_resp_df.to_json(f, orient="records", lines=True)

    for epoch in range(n_epochs):
        print(f"Epoch {epoch}")

        if lr_sched is not None:
            lr = float(lr_sched(epoch))
            print(f"Setting LR to {lr}")

            for param_group in optimizer.param_groups:
                param_group["lr"] = lr

        train_ds = get_train_ds(epoch)

        if len(train_ds) > 0:
            train_dl = DataLoader(train_ds, batch_size=train_batch_size, shuffle=True)
            train_resp_df = run_critique_train_epoch(
                model,
                tokenizer,
                train_dl,
                optimizer,
                fixed_sizes=fixed_sizes,
                inference_dtype=inference_dtype,
                max_ctx=max_ctx,
            )

            print("training accuracy")
            train_results = get_aggr_train_res(train_resp_df)
            has_trained = True
        else:
            has_trained = False

        fin_epoch = epoch + 1

        if eval_every is None or (fin_epoch % eval_every == 0):
            # eval_resp_df = await run_critique_eval(
            #     model, tokenizer, val_dl, inference_dtype=inference_dtype, fixed_sizes=fixed_sizes, max_ctx=max_ctx
            # )
            model.save_pretrained(tmp_save_dir)
            to_device(model, optimizer, "cpu")
            eval_resp_df = await run_critique_eval_vllm(
                model_name=tmp_save_dir,
                tokenizer_name=tokenizer.name_or_path,
                test_ds=val_ds,
                temperature=0,
                max_ctx=max_ctx,
            )
            to_device(model, optimizer, "cuda")
            shutil.rmtree(tmp_save_dir, ignore_errors=True)

            print("validation accuracy")
            eval_results = get_aggr_val_res(eval_resp_df)

            if has_trained:
                eval_results["train"] = train_results

            all_eval_results.append([fin_epoch, eval_results])

            if base_save_dir is not None:
                with open(f"{base_save_dir}/res_epoch_{fin_epoch}.json", "w") as f:
                    json.dump(eval_results, f, indent=4)
                with open(f"{base_save_dir}/res_df_epoch_{fin_epoch}.jsonl", "w") as f:
                    eval_resp_df.to_json(f, orient="records", lines=True, default_handler=str)

                if has_trained:
                    with open(f"{base_save_dir}/res_train_df_epoch_{fin_epoch}.jsonl", "w") as f:
                        train_resp_df.to_json(f, orient="records", lines=True, default_handler=str)
                else:
                    with open(f"{base_save_dir}/res_train_df_epoch_{fin_epoch}.nodata", "w") as f:
                        f.write("no data")
        if base_save_dir is not None:
            assert save_model_name is not None
            d = f"{base_save_dir}/weights_epoch_{fin_epoch}"
            if save_weights_every == False:
                if fin_epoch == n_epochs:
                    Path(d).mkdir(parents=True, exist_ok=True)
            else:
                if (fin_epoch % save_weights_every == 0) or (fin_epoch == n_epochs):
                    model.save_pretrained(d)

    if base_save_dir is not None:
        with open(f"{base_save_dir}/res_all.json", "w") as f:
            json.dump(all_eval_results, f)

    # remove tmp
    shutil.rmtree(tmp_save_dir, ignore_errors=True)

    return all_eval_results


def cosine_decay_fn_raw(current_step, start_lr, end_lr, decay_steps):
    """
    Calculates the learning rate at a given step using a cosine decay.

    :param current_step: The current step (e.g., epoch or batch number).
    :return: The decayed learning rate.
    """
    if current_step >= decay_steps:
        return end_lr  # After decay_steps, return the minimum learning rate
    else:
        cosine_decay = 0.5 * (1 + math.cos(math.pi * current_step / decay_steps))
        decayed_lr = (start_lr - end_lr) * cosine_decay + end_lr
        return decayed_lr


def create_cosine_decay_fn(start_lr, end_lr, decay_steps):
    """
    Creates a cosine decay function with the given parameters.

    :param start_lr: The initial learning rate.
    :param decay_steps: The number of steps over which to decay the learning rate.
    :param end_lr: The minimum learning rate to return after decay is complete.
    :return: A function that takes the current step and returns the decayed learning rate.
    """

    return partial(cosine_decay_fn_raw, start_lr=start_lr, end_lr=end_lr, decay_steps=decay_steps)
