import math
from functools import partial

import numpy as np


def cosine_decay_fn_raw(current_step, start_lr, end_lr, decay_steps):
    """
    Calculates the learning rate at a given step using a cosine decay.

    :param current_step: The current step (e.g., epoch or batch number).
    :return: The decayed learning rate.
    """
    if current_step >= decay_steps:
        return end_lr  # After decay_steps, return the minimum learning rate
    else:
        cosine_decay = 0.5 * (1 + math.cos(math.pi * current_step / decay_steps))
        decayed_lr = (start_lr - end_lr) * cosine_decay + end_lr
        return decayed_lr


def create_cosine_decay_fn(start_lr, end_lr, decay_steps):
    """
    Creates a cosine decay function with the given parameters.

    :param start_lr: The initial learning rate.
    :param decay_steps: The number of steps over which to decay the learning rate.
    :param end_lr: The minimum learning rate to return after decay is complete.
    :return: A function that takes the current step and returns the decayed learning rate.
    """

    return partial(cosine_decay_fn_raw, start_lr=start_lr, end_lr=end_lr, decay_steps=decay_steps)


def ramp_fn_raw(current_step, start_ramp_lr, normal_start_lr, ramp_steps, rest_fn):
    """
    Calculates the learning rate at a given step using a ramp decay.

    :param current_step: The current step (e.g., epoch or batch number).
    :return: The decayed learning rate.
    """
    if current_step < ramp_steps:
        return float(np.interp(current_step, [0, ramp_steps], [start_ramp_lr, normal_start_lr]))
    else:
        return rest_fn(current_step)


def create_ramp_fn(start_ramp_lr, normal_start_lr, ramp_steps, rest_fn):
    return partial(
        ramp_fn_raw,
        start_ramp_lr=start_ramp_lr,
        normal_start_lr=normal_start_lr,
        ramp_steps=ramp_steps,
        rest_fn=rest_fn,
    )
