import asyncio
import json
import os
from typing import Optional

import nest_asyncio
import numpy as np
import openai
import tiktoken

import sandbagging.utils.optional as op
from sandbagging.extract_math_answer import extract_math_answer, quick_convert_to_ascii_math_helper
from sandbagging.math_data_basic_setup import all_math
from sandbagging.math_eval_answer import eval_math

nest_asyncio.apply()

from sandbagging.utils.llm_api.chatlib import Chat
from sandbagging.all_inference import model_api

# %%

if "OPENAI_API_KEY" not in os.environ:
    openai.api_key_path = os.path.expanduser("~/.openai_api_key")
    try:
        with open(openai.api_key_path, "r") as f:
            os.environ["OPENAI_API_KEY"] = f.read().strip()
    except FileNotFoundError:
        ...


# %%

problem_solution_convert_to_spanish = """Please translate the below math problem and math solution to Spanish.

<problem>
{problem}
</problem>

<solution>
{solution}
</solution>

Return the translated problem in <translated_problem> XML tags. Return the translated solution in <translated_solution> XML tags."""

ENGLISH_problem_solution_in_spanish_convert_to_ascii_math_format = """Please convert the below math problem and math solution from TeX to AsciiMath. Use ` as the delimiter for AsciiMath.

<problem>
{problem}
</problem>

<solution>
{solution}
</solution>

Return the converted problem in <converted_problem> XML tags. Return the converted solution in <converted_solution> XML tags. Keep the problem and solution in Spanish."""
# When you do the conversion, wrap the final answer like ▶`ANSWER`◀. This is so that it can be automatically extracted from the output. (If the final answer is surrounded like \\boxed{ANSWER} in the TeX, then you should convert this \\boxed{ANSWER} to ▶`ANSWER`◀.)

problem_solution_in_spanish_convert_to_ascii_math_format = """Por favor, traduce el siguiente problema matemático y la solución matemática de TeX a AsciiMath. Usa ` como delimitador para AsciiMath.

<problem>
{problem}
</problem>

<solution>
{solution}
</solution>

Devuelve el problema convertido en etiquetas XML <converted_problem>. Devuelve la solución convertida en etiquetas XML <converted_solution>. Mantén el problema y la solución en español."""

# %%


def remove_boxed(latex_code):
    """
    Remove the \boxed{} command from LaTeX code, handling nested curly braces.

    Args:
    - latex_code (str): The input LaTeX code string.

    Returns:
    - str: The LaTeX code with \boxed{} removed, preserving nested content.
    """
    result = ""
    i = 0
    while i < len(latex_code):
        if latex_code[i : i + 7] == "\\boxed{":
            depth = 1
            content = ""
            i += 7  # Skip past \boxed{
            while i < len(latex_code) and depth > 0:
                if latex_code[i] == "{":
                    depth += 1
                    if depth > 1:  # Preserve nested braces
                        content += "{"
                elif latex_code[i] == "}":
                    depth -= 1
                    if depth > 0:  # Preserve nested braces
                        content += "}"
                else:
                    content += latex_code[i]
                i += 1
            result += content  # Append the content without the outermost braces
        else:
            result += latex_code[i]
            i += 1
    return result


# %%

assert remove_boxed(r"\boxed{XYZ}") == "XYZ", "Test 1 Failed: Simple case"
assert remove_boxed(r"\boxed{\frac{1}{2}}") == r"\frac{1}{2}", "Test 2 Failed: Inside frac"
assert (
    remove_boxed(r"Before \boxed{content} after.") == "Before content after."
), "Test 4 Failed: Boxed with surrounding text"
assert remove_boxed(r"\boxed{\alpha} + \boxed{\beta}") == r"\alpha + \beta", "Test 6 Failed: Multiple boxed"
assert remove_boxed(r"ASD\boxed{\alpha} + \boxed{\beta}") == r"ASD\alpha + \beta", "Test 6 Failed: Multiple boxed"
assert remove_boxed(r"ASD\boxed{\alpha} + \boxed{\beta}SDI") == r"ASD\alpha + \betaSDI", "Test 6 Failed: Multiple boxed"
assert remove_boxed(r"No boxed content here.") == "No boxed content here.", "Test 7 Failed: No boxed"
assert remove_boxed(r"\boxed{a{b}c}") == "a{b}c", "Test 8 Failed: Curly braces inside boxed"

# %%

gpt_4_tok = tiktoken.encoding_for_model("gpt-4-turbo-preview")

# %%


async def fully_translate_problem_solution(
    problem: str, solution: str
) -> tuple[Optional[str], Optional[str], tuple[str, str], float]:
    chat = Chat(model_api)

    chat.user_say(problem_solution_convert_to_spanish.format(problem=problem.strip(), solution=solution.strip()))
    prompt_tok_count = chat.current_tokens()
    out_trans = await chat.ask_models(["gpt-4-turbo-preview", "gpt-4-0125-preview"], t=0.0, parse=True)
    completion_tok_count = len(gpt_4_tok.encode(out_trans.text))
    translated_problem = op.unwrap(out_trans.find("translated_problem")).text.strip()
    translated_solution = op.unwrap(out_trans.find("translated_solution")).text.strip()

    before_translated_solution = translated_solution
    translated_solution = remove_boxed(translated_solution)

    assert "boxed" not in translated_solution
    if "boxed" not in before_translated_solution:
        assert translated_solution == before_translated_solution

    final_problem = None
    final_solution = None
    iters = 0
    while (final_problem is None or final_solution is None) and iters < 4:
        if iters >= 1:
            print("iters:", iters)
        chat_conv = Chat(model_api)

        chat_conv.user_say(
            problem_solution_in_spanish_convert_to_ascii_math_format.format(
                problem=translated_problem, solution=translated_solution
            )
        )
        prompt_tok_count += chat_conv.current_tokens()
        out_conv = await chat_conv.ask_models(
            ["gpt-4-turbo-preview", "gpt-4-0125-preview"], t=1.0 if iters > 1 else 0.0, parse=True, resample=iters >= 2
        )
        completion_tok_count = len(gpt_4_tok.encode(out_conv.text))

        final_problem = op.map(out_conv.find("converted_problem"), lambda x: x.text.strip())
        final_solution = op.map(out_conv.find("converted_solution"), lambda x: x.text.strip())
        iters += 1

    assert final_problem is not None
    assert final_solution is not None

    final_problem = quick_convert_to_ascii_math_helper(final_problem)
    final_solution = quick_convert_to_ascii_math_helper(final_solution)

    return (
        final_problem,
        final_solution,
        (translated_problem, translated_solution),
        (prompt_tok_count / 1000 * 0.01 + completion_tok_count / 1000 * 0.03),
    )


# %%

with open("math_outputs/deepseek_math_inference/inference_outputs.json", "r") as f:
    deepseek_math_inference = json.load(f)

# %%

# assert sum("boxed" in x["output"] for x in deepseek_math_inference) > 1000
# deepseek_math_inference_box_removed = [
#     dict(problem=x["problem"], output=remove_boxed(x["output"])) for x in deepseek_math_inference
# ]
# assert all("boxed" not in x["output"] for x in deepseek_math_inference_box_removed)

# %%


async def run_all():
    problem_solutions = [(x["problem"], x["output"]) for x in deepseek_math_inference]

    total_cost = 0.0
    last_mul = 0

    async def translate_report_spend(problem: str, solution: str):
        nonlocal total_cost, last_mul
        out = await fully_translate_problem_solution(problem, solution)
        total_cost += out[-1]
        if total_cost > last_mul * 10:
            print(total_cost)
            last_mul += 1

        return out

    out = await asyncio.gather(*[translate_report_spend(*item) for item in problem_solutions])

    print("final cost:", total_cost)

    out_dict_lst = [
        dict(
            problem=problem,
            output=output,
            fully_translated_problem=fully_translated_problem,
            fully_translated_solution=fully_translated_solution,
            in_spanish_problem=in_spanish_problem,
            in_spanish_solution=in_spanish_solution,
        )
        for (problem, output), (
            fully_translated_problem,
            fully_translated_solution,
            (in_spanish_problem, in_spanish_solution),
            _,
        ) in zip(problem_solutions, out)
    ]

    return out_dict_lst


# %%

all_outputs = await run_all()

len(all_outputs)

# %%


problem_to_idx = {x: i for i, x in enumerate(all_math["problem"].values)}

# %%

preds = [extract_math_answer(o["problem"], o["output"]) for o in all_outputs]
extracted_answers = [all_math["extracted_answer"].values[problem_to_idx[o["problem"]]] for o in all_outputs]
corr = np.array([eval_math(p, a) for p, a in zip(preds, extracted_answers)])

corr.mean()

# %%

preds = [
    extract_math_answer(o["problem"], o["fully_translated_solution"], spanish_and_ascii_model=True) for o in all_outputs
]
extracted_answers = [all_math["extracted_answer"].values[problem_to_idx[o["problem"]]] for o in all_outputs]
corr_trans = np.array([eval_math(p, a) for p, a in zip(preds, extracted_answers)])

corr_trans.mean()

# %%

items = (corr & (~corr_trans)).nonzero()[0]
len(items)
items

# %%

# idx = 8513

# o  = all_outputs[idx]
# pred = extract_math_answer(o["problem"], o["fully_translated_solution"], spanish_and_ascii_model=True)
# extracted_answer = all_math["extracted_answer"].values[problem_to_idx[o["problem"]]]

# print(o["fully_translated_solution"])

# print(pred)
# print(extracted_answer)

# %%


with open("math_outputs/translated_outputs.json", "w") as f:
    json.dump(all_outputs, f)

# %%
