from __future__ import annotations

import json
import keyword
import os
import re
import sys
import platform
from ast import literal_eval
from collections import defaultdict
from itertools import islice
from pathlib import Path
from typing import Any, Optional, Tuple, Union
from typing import Sequence, TypeVar

import attrs
from attrs import field
from attrs.validators import instance_of
from cattrs.preconf.json import make_converter


def is_list_of_str(inst, attr, val):
    if not (isinstance(val, list) and all(isinstance(x, str) for x in val)):
        raise TypeError(f"{attr.name} must be a list of strings")


def str_is_valid_var(val):
    return isinstance(val, str) and val.isidentifier() and not keyword.iskeyword(val)


def is_valid_var(inst, attr, val):
    if not str_is_valid_var(val):
        raise TypeError(f"{attr.name} must be a valid variable name, got {val!r}")


TestCaseId = tuple[str, str, str]


@attrs.frozen
class InputOutputTestCases:
    inputs: list[str] = field(validator=is_list_of_str)
    outputs: list[str] = field(validator=is_list_of_str)

    def example_inputs(self):
        return self.inputs

    @property
    def n(self):
        return len(self.inputs)

    @property
    def ids(self) -> list[TestCaseId]:
        return [("", x, y) for x, y in zip(self.inputs, self.outputs)]

    def add(self, other: InputOutputTestCases):
        assert len(other.outputs) == len(other.inputs)
        assert len(self.outputs) == len(self.inputs)

        return InputOutputTestCases(
            inputs=self.inputs + other.inputs,
            outputs=self.outputs + other.outputs,
        )

    def add_unwrap(self, other: TestCases):
        assert isinstance(other, InputOutputTestCases)
        return self.add(other)

    def remove(self, test_idxs: list[int]):
        return InputOutputTestCases(
            inputs=[x for i, x in enumerate(self.inputs) if i not in test_idxs],
            outputs=[x for i, x in enumerate(self.outputs) if i not in test_idxs],
        )


@attrs.frozen
class ToEval:
    code: str = field(validator=instance_of(str))


@attrs.frozen
class PythonTestCases:
    is_solution_class: bool = field(validator=instance_of(bool))
    fn_name: str = field(validator=is_valid_var)
    inputs: list[tuple[Union[Any, ToEval], ...]]
    outputs: list[Union[ToEval, Any]]

    def arg_strings(self) -> list[str]:
        return [f"({', '.join(e.code if isinstance(e, ToEval) else repr(e) for e in x)})" for x in self.inputs]

    def example_inputs(self) -> list[str]:
        return self.arg_strings()

    def boolean_statements(self, max_nb_examples: Optional[int] = None) -> list[str]:
        assert len(self.inputs) == len(self.outputs), f"{len(self.inputs)} != {len(self.outputs)}"

        if max_nb_examples is None:
            max_nb_examples = len(self.inputs)
        return [
            f"{self.fn_name}{x} == {y.code if isinstance(y, ToEval) else repr(y)}"
            for x, y in islice(zip(self.arg_strings(), self.outputs), max_nb_examples)
        ]

    def get_assertions(self):
        assert len(self.inputs) == len(self.outputs), f"{len(self.inputs)} != {len(self.outputs)}"
        return [
            f"assert ({self.fn_name}{x} == {y.code if isinstance(y, ToEval) else repr(y)})"
            for x, y in zip(self.arg_strings(), self.outputs)
        ]

    def explanation(self, max_nb_examples: Optional[int] = 2) -> str:
        """Return an explanation of the task using the data from the test cases.

        The explanation is in the following format:
        You should write a function named second_biggest that takes 1 input.
        Examples:
        assert (second_biggest([1, 2, 3]) == 2)
        assert (second_biggest([4, 3, 2, 1]) == 3)
        """
        if self.is_solution_class:
            raise NotImplementedError("Solution classes are not supported yet.")

        plural = "s" if len(self.inputs[0]) > 1 else ""

        lines = [f"You should write a function named {self.fn_name} that takes {len(self.inputs[0])} input{plural}."]
        if max_nb_examples:
            lines.append("Examples:")
            lines.extend(self.get_assertions()[:max_nb_examples])

        return "\n".join(lines)

    @property
    def n(self) -> int:
        return len(self.inputs)

    @property
    def ids(self) -> list[TestCaseId]:
        if self.is_solution_class:
            raise NotImplementedError("Solution classes are not supported yet.")
        # return [hash((self.fn_name, repr(x), repr(y))) for x, y in zip(self.inputs, self.outputs)]
        # return [(self.fn_name, repr(x), repr(y)) for x, y in zip(self.inputs, self.outputs)]
        return [(self.fn_name, repr(x), repr(y)) for x, y in zip(self.inputs, self.outputs)]

    def add(self, other: PythonTestCases) -> PythonTestCases:
        assert len(other.outputs) == len(other.inputs)
        assert len(self.outputs) == len(self.inputs)
        assert self.fn_name == other.fn_name

        return PythonTestCases(
            is_solution_class=self.is_solution_class,
            fn_name=self.fn_name,
            inputs=self.inputs + other.inputs,
            outputs=self.outputs + other.outputs,
        )

    def add_unwrap(self, other: TestCases) -> PythonTestCases:
        assert isinstance(other, PythonTestCases)
        return self.add(other)

    def remove(self, test_idxs: list[int]) -> PythonTestCases:
        return PythonTestCases(
            is_solution_class=self.is_solution_class,
            fn_name=self.fn_name,
            inputs=[x for i, x in enumerate(self.inputs) if i not in test_idxs],
            outputs=[x for i, x in enumerate(self.outputs) if i not in test_idxs],
        )


TestCases = Union[InputOutputTestCases, PythonTestCases]


@attrs.frozen
class LoadedProblem:
    task_id: int = field(validator=instance_of(int))
    dataset: str = field(validator=instance_of(str))
    description: str = field(validator=instance_of(str))
    solutions: list[str] = field(validator=is_list_of_str)
    base_code: str = field(validator=instance_of(str))
    base_code_fmt_example: str = field(validator=instance_of(str))
    difficulty: str = field(validator=instance_of(str))
    test_cases: TestCases
    meta_data: dict[str, str] = {}
