from __future__ import annotations

import datetime
import os
import socket
import subprocess
import sys
import time
from contextlib import closing
from dataclasses import dataclass
from typing import Any, Callable, Optional

import boto3  # type: ignore
import fabric
import invoke
import numpy
from multiprocess import Process
from rpyc import async_, classic
from typing_extensions import Protocol

ec2 = boto3.client("ec2", region_name=os.environ.get("AWS_REGION", "us-west-2"))


def keyfile() -> str:
    return os.environ["RR_DEV_KEY_FILE"]


def choose_open_port() -> int:
    # This is not guaranteed to work; e.g. we release the socket before returning.
    # But the fabric API is not very amenable to better methods (e.g. it doesn't
    # let you pass your own socket to the forward_local method)
    with closing(socket.socket(socket.AF_INET, socket.SOCK_STREAM)) as s:
        s.bind(("localhost", 0))
        s.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        return s.getsockname()[1]


class InstanceHandle(Protocol):
    name: str
    instance_id: str

    def run_command(self, cmd: str, asynchronous=True, pty=False) -> invoke.runners.Result:
        ...

    def run_python(self, f, asynchronous: bool = False) -> Any:
        ...

    def ready_for_commands(self) -> bool:
        ...

    def wait_for_ready(self, timeout: Optional[float]):
        ...

    def send_dir(self, local_path: str, remote_path: str, exclude=".git", filter=":- .gitignore") -> subprocess.Popen:
        ...


class EC2InstanceHandle:
    def __init__(self, name: str, instance_id: str, address: Optional[str], rpyc_port: Optional[int], remote_py: Any):
        self.name = name
        self.instance_id = instance_id
        self.address = address
        self.rpyc_port = rpyc_port
        self.remote_py = remote_py

    @staticmethod
    def from_boto3(description: dict):
        name: str = [t["Value"] for t in description["Tags"] if t["Key"] == "Name"][0]
        return EC2InstanceHandle(
            name=name,
            instance_id=description["InstanceId"],
            address=description.get("PublicIpAddress"),
            remote_py=None,
            rpyc_port=None,
        )

    def query_address(self) -> Optional[str]:
        description = ec2.describe_instances(InstanceIds=[self.instance_id])
        description = ec2.describe_instances(InstanceIds=[self.instance_id])
        for res in description["Reservations"]:
            for i in res["Instances"]:
                if i["InstanceId"] == self.instance_id:
                    return i.get("PublicIpAddress")
        return None

    def port_open(self, port: int) -> bool:
        test_socket = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        location = (self.address, port)
        connection_result = test_socket.connect_ex(location)
        return connection_result >= 0

    def ready_for_commands(self) -> bool:
        address = self.query_address()
        if address is None:
            return False
        self.address = address
        if not self.port_open(22):
            return False
        return True

    def wait_for_ready(self, timeout: Optional[float] = None):
        start_time = datetime.datetime.now()
        while self.address is None:
            if timeout is not None and datetime.datetime.now() > start_time + datetime.timedelta(timeout):
                raise IOError("Exceeded AWS timeout while waiting for addresses")
            address = self.query_address()
            if address is None:
                time.sleep(1)
            self.address = address

        # we have an address now
        while not self.port_open(22):
            if timeout is not None and datetime.datetime.now() > start_time + datetime.timedelta(timeout):
                raise IOError("Exceeded AWS timeout while waiting for open port")
            time.sleep(1)

        self.start_rpyc()

    def connect(self) -> fabric.Connection:
        config = fabric.Config()
        return fabric.Connection(
            self.address,
            user="ubuntu",
            config=config,
            connect_kwargs={
                "gss_host": self.instance_id,
                "key_filename": keyfile(),
            },
        )

    def run_command(self, cmd: str, asynchronous=True, pty=False) -> fabric.Result:
        c = self.connect()
        return c.run(
            cmd, asynchronous=asynchronous, pty=pty, hide=False, echo=True, out_stream=sys.stdout, err_stream=sys.stderr
        )

    def start_rpyc(self):
        self.rpyc_port = choose_open_port()
        self.run_command(
            f"screen -d -m rpyc_classic.py -p {self.rpyc_port} -m forking; sleep 1",
            asynchronous=False,
            pty=False,
        )

    def run_python(self, f, asynchronous: bool = False) -> Any:
        with self.connect().forward_local(self.rpyc_port):
            rpy = classic.connect("localhost", port=self.rpyc_port)
            if asynchronous:
                # The connection is closed after this, so the result doesn't allow you to extract any info about how it
                # went.
                async_(rpy.teleport(f))()
            else:
                return classic.obtain(rpy.teleport(f)())

    def send_dir(self, local_path: str, remote_path: str, exclude=".git", filter=":- .gitignore") -> subprocess.Popen:
        if local_path.endswith("/"):
            local_path = local_path[:-1]
        if remote_path.endswith("/"):
            remote_path = remote_path[:-1]

        return subprocess.Popen(
            [
                "rsync",
                f"--exclude={exclude}",
                f"--filter={filter}",
                "-rvtz",
                "-e",
                (
                    "ssh "
                    + "-o CheckHostIP=no "
                    + f"-o HostKeyAlias={self.instance_id} "
                    + "-o StrictHostKeyChecking=no "
                    + f"-i {keyfile()}"
                ),
                f"{local_path}/",
                f"ubuntu@{self.address}:{remote_path}",
            ],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
        )


class LocalInstanceHandle:
    def __init__(self, name: str, instance_id: str):
        self.name = name
        self.instance_id = instance_id

    def wait_for_ready(self, timeout: Optional[float] = None):
        if timeout is not None and timeout < 0:
            return False
        return True

    def run_command(self, cmd: str, asynchronous: bool = True, pty: bool = False) -> invoke.runners.Result:
        runner = invoke.runners.Local(invoke.context.Context())
        result = runner.run(
            cmd, asynchronous=asynchronous, pty=pty, hide=False, echo=True, out_stream=sys.stdout, err_stream=sys.stderr
        )
        assert type(result) is invoke.runners.Result or type(result) is invoke.runners.Promise
        return result

    def run_python(self, f, asynchronous=False):
        if asynchronous:
            Process(target=f).start()
        else:
            return f()

    def ready_for_commands(self) -> bool:
        return True

    def send_dir(self, local_path: str, remote_path: str, exclude=".git", filter=":- .gitignore") -> subprocess.Popen:
        if local_path.endswith("/"):
            local_path = local_path[:-1]
        if remote_path.endswith("/"):
            remote_path = remote_path[:-1]

        return subprocess.Popen(
            [
                "rsync",
                f"--exclude={exclude}",
                f"--filter={filter}",
                "-rvtz",
                f"{local_path}/",
                remote_path,
            ],
            stdout=subprocess.PIPE,
            stderr=subprocess.PIPE,
        )


@dataclass
class ClusterHandle:
    n: int
    template: str
    identifier: str
    name: str
    instances: list[InstanceHandle]
    local: bool

    def __init__(self, n: int = 1, template: str = "template", identifier: str = "auto", local: bool = False):
        self.local = local
        self.n = n or 1
        self.template = template
        self.identifier = identifier
        self.instances = []
        self.name = "org-" + os.environ["USER"] + "-" + self.identifier + "-" + datetime.datetime.now().isoformat()

    def start_instances(self):
        if self.local:
            self.instances = [LocalInstanceHandle(self.name, self.name + str(i)) for i in range(self.n)]
        else:
            result = ec2.run_instances(
                LaunchTemplate={"LaunchTemplateName": self.template},
                TagSpecifications=[
                    {
                        "ResourceType": "instance",
                        "Tags": [
                            {"Key": "Name", "Value": self.name},
                            {"Key": "User", "Value": os.environ["USER"]},
                            {"Key": "Category", "Value": "auto"},
                        ],
                    }
                ],
                MaxCount=self.n,
                MinCount=self.n,
            )

            self.instances = [EC2InstanceHandle.from_boto3(i) for i in result["Instances"]]

    def terminate(self):
        return ec2.terminate_instances(InstanceIds=[i.instance_id for i in self.instances])

    def stop(self):
        return ec2.stop_instances(InstanceIds=[i.instance_id for i in self.instances])

    def wait_for_ready(self, timeout: float = 120.0):
        for i in self.instances:
            i.wait_for_ready(timeout=timeout)

    @staticmethod
    def new_running(
        n: int = 1, template: str = "template", identifier: str = "auto", local: bool = False
    ) -> ClusterHandle:
        cluster = ClusterHandle(n=n, template=template, identifier=identifier, local=local)
        cluster.start_instances()
        present = False
        while not present:
            try:
                cluster.wait_for_ready()
                present = True
            except:
                time.sleep(1)
        return cluster

    def send_dir(self, local_path: str, remote_path: str, exclude=".git", filter=":- .gitignore", blocking=True):
        sps = []
        for instance in self.instances:
            sps.append(instance.send_dir(local_path, remote_path, exclude=exclude, filter=filter))
        if blocking:
            for sp in sps:
                sp.wait()
                if sp.returncode != 0:
                    raise IOError("rsync failed")
        return sps

    def run_command(self, cmd: str, block=False) -> list[fabric.Result]:
        rs = []
        for instance in self.instances:
            rs.append(instance.run_command(cmd, asynchronous=True))
        if block:
            result = []
            for r in rs:
                result.append(r.join())
            return result
        else:
            return rs

    def run_python(self, f: Callable[[], Any], asynchronous=False) -> list[Any]:
        # Asynchronous execution just returns a list of Nones
        rs = []
        for instance in self.instances:
            rs.append(instance.run_python(f, asynchronous=asynchronous))

        return rs
