from functools import cache
from typing import Literal, get_args

import attrs
import torch
from transformers import AutoModelForCausalLM, AutoTokenizer
from vllm import LLM
from vllm.core.scheduler import logger as scheduler_logger
from vllm.engine.llm_engine import logger as llm_engine_logger
from vllm.model_executor.layers.sampler import Sampler
from vllm.model_executor.weight_utils import logger as weight_utils_logger
from vllm.worker.model_runner import logger as model_runner_logger

from sandbagging.dataset_wrapping import cache_dir
from sandbagging.setting_types import SettingKind

scheduler_logger.setLevel("ERROR")
model_runner_logger.setLevel("ERROR")
weight_utils_logger.setLevel("ERROR")
llm_engine_logger.setLevel("ERROR")

TESTING_SMALL_MODEL = "facebook/opt-125m"

MISTRAL_MODEL = "mistralai/Mistral-7B-v0.1"
OLD_LLAMA_MODEL = "chavinlo/alpaca-native"

INTERNET_LOAD_NAMES = [TESTING_SMALL_MODEL, MISTRAL_MODEL, OLD_LLAMA_MODEL]

BASELINE_TRAINED_MISTRAL = "mmlu_models/baseline_mmlu_mistral/baseline_mmlu_mistral_epoch_8"
BASELINE_TRAINED_OLD_LLAMA = "mmlu_models/baseline_mmlu_old_llama/weights_epoch_2"

device = "cuda:0"  # the device to load the model onto

weights_dtype = torch.float16
inference_dtype = torch.float16
inference_dtype_str = "float16"

DEFAULT_MAX_CTX = 1536


@attrs.frozen
class SizeInfo:
    parameters: float
    standard_lr_val: float = 1.5e-6
    extra_mcqa_lr_mul: float = 1.0

    @property
    def standard_lr_by_setting(self) -> dict[SettingKind, float]:
        DEFAULT_MCQA_LR_MUL = 1.0
        return {
            "code": self.standard_lr_val,
            "critique": self.standard_lr_val,
            "ntp": self.standard_lr_val,  # ?
            "math": self.standard_lr_val,
            "math_spanish_ascii": self.standard_lr_val,
            "wmdp": self.standard_lr_val * DEFAULT_MCQA_LR_MUL * self.extra_mcqa_lr_mul,
            "mmlu": self.standard_lr_val * DEFAULT_MCQA_LR_MUL * self.extra_mcqa_lr_mul,
            "hellaswag": self.standard_lr_val * DEFAULT_MCQA_LR_MUL * self.extra_mcqa_lr_mul,
        }


SIZE_INFOS = {
    "7B": SizeInfo(6.7e9),
    "3B": SizeInfo(3e9),
    "1B": SizeInfo(1e9),
    "400M": SizeInfo(4e8),
    "160M": SizeInfo(1.6e8),
    "70M": SizeInfo(7e7),
    "125M": SizeInfo(1.25e8),
}


@attrs.frozen
class BaseModelInfo:
    full_name: str  # path or internet name
    size: SizeInfo


PYTHIA70M = "pythia70m"
PYTHIA400M = "pythia400m"
PYTHIA1B = "pythia1b"
PYTHIA3B = "pythia3b"
PYTHIA7B = "pythia7b"
UNIFORM = "uniform"
UNIFORMPP = "uniformpp"
FIXEDRDM = "fixedrdm"
STABLE_ZEPH2B = "zephyr2b"
STABLE_ZEPH3B = "zephyr3b"
MISTRAL7B = "mistral7b"
OLD_LLAMA7B = "old_llama7b"
DEEPSEEK_MATH = "deepseek_math7b"
DEEPSEEK = "deepseek7b"
CODELLAMA7B = "codellama7b"
PYTHIA160M = "pythia160m"
OPT125M = "opt125m"
CGPT2 = "cgpt2"

CGPT2_PATH = f"{cache_dir}/{CGPT2}/locked/weights_epoch_0"
SGPT2_PATH = f"{cache_dir}/{CGPT2}/good/weights_epoch_0"

TINY_LLAMA1B = "tiny_llama1b"

PYTHIA_EXTRA_MCQA_LR_MUL = 0.75

MODEL_INFOS = {
    DEEPSEEK: BaseModelInfo(
        "deepseek-ai/deepseek-coder-6.7b-base", attrs.evolve(SIZE_INFOS["7B"], standard_lr_val=1.5e-6)
    ),
    DEEPSEEK_MATH: BaseModelInfo(
        "deepseek-ai/deepseek-math-7b-rl", attrs.evolve(SIZE_INFOS["7B"], standard_lr_val=1.5e-6)
    ),
    CODELLAMA7B: BaseModelInfo("codellama/CodeLlama-7b-hf", SIZE_INFOS["7B"]),
    MISTRAL7B: BaseModelInfo("mistralai/Mistral-7B-v0.1", attrs.evolve(SIZE_INFOS["7B"], standard_lr_val=1.5e-7)),
    STABLE_ZEPH2B: BaseModelInfo(
        "stabilityai/stablelm-2-zephyr-1_6b", attrs.evolve(SIZE_INFOS["3B"], standard_lr_val=3e-7)
    ),
    STABLE_ZEPH3B: BaseModelInfo(
        "stabilityai/stablelm-zephyr-3b", attrs.evolve(SIZE_INFOS["3B"], standard_lr_val=3e-7)
    ),
    PYTHIA160M: BaseModelInfo("EleutherAI/pythia-160m-deduped", SIZE_INFOS["160M"]),
    OPT125M: BaseModelInfo("facebook/opt-125m", SIZE_INFOS["125M"]),
    CGPT2: BaseModelInfo(CGPT2_PATH, SIZE_INFOS["125M"]),
    UNIFORM: BaseModelInfo(
        f"{cache_dir}/pythia-uniform", attrs.evolve(SIZE_INFOS["70M"], standard_lr_val=1e-20)
    ),  # low lr to prevent training
    UNIFORMPP: BaseModelInfo(
        f"{cache_dir}/pythia-uniform", attrs.evolve(SIZE_INFOS["70M"], standard_lr_val=1e-20)
    ),  # low lr to prevent training
    FIXEDRDM: BaseModelInfo(
        f"{cache_dir}/pythia-uniform", attrs.evolve(SIZE_INFOS["70M"], standard_lr_val=1e-20)
    ),  # low lr to prevent training
    PYTHIA70M: BaseModelInfo(
        "EleutherAI/pythia-70m-deduped",
        attrs.evolve(SIZE_INFOS["70M"], standard_lr_val=2e-6, extra_mcqa_lr_mul=PYTHIA_EXTRA_MCQA_LR_MUL),
    ),
    PYTHIA400M: BaseModelInfo(
        "EleutherAI/pythia-410m-deduped",
        attrs.evolve(SIZE_INFOS["400M"], standard_lr_val=2e-6, extra_mcqa_lr_mul=PYTHIA_EXTRA_MCQA_LR_MUL),
    ),
    PYTHIA1B: BaseModelInfo(
        "EleutherAI/pythia-1b-deduped",
        attrs.evolve(SIZE_INFOS["1B"], standard_lr_val=1.6e-6, extra_mcqa_lr_mul=PYTHIA_EXTRA_MCQA_LR_MUL),
    ),
    PYTHIA3B: BaseModelInfo(
        "EleutherAI/pythia-2.8b-deduped",
        attrs.evolve(SIZE_INFOS["3B"], standard_lr_val=1.25e-6, extra_mcqa_lr_mul=PYTHIA_EXTRA_MCQA_LR_MUL),
    ),
    PYTHIA7B: BaseModelInfo(
        "EleutherAI/pythia-6.9b-deduped",
        attrs.evolve(SIZE_INFOS["7B"], standard_lr_val=1e-6, extra_mcqa_lr_mul=PYTHIA_EXTRA_MCQA_LR_MUL),
    ),
    OPT125M: BaseModelInfo("facebook/opt-125m", SIZE_INFOS["125M"]),
    OLD_LLAMA7B: BaseModelInfo("chavinlo/alpaca-native", attrs.evolve(SIZE_INFOS["7B"], standard_lr_val=1e-6)),
    TINY_LLAMA1B: BaseModelInfo(
        "TinyLlama/TinyLlama-1.1B-Chat-v1.0", attrs.evolve(SIZE_INFOS["1B"], standard_lr_val=2e-6)
    ),
}


@cache
def get_eos_token(tokenizer_name: str):
    return AutoTokenizer.from_pretrained(tokenizer_name).eos_token


@cache
def get_tokenizer_len(tokenizer_name: str):
    return len(AutoTokenizer.from_pretrained(tokenizer_name))


def load_tokenizer(name_or_path: str):
    tok = AutoTokenizer.from_pretrained(name_or_path, trust_remote_code=True)
    # We also use prepare_inputs_for_generation.
    tok.pad_token = tok.eos_token
    tok.padding_side = "left"
    tok.truncation_side = "left"
    return tok


def load_model_tokenizer(
    load_model_path, tokenizer_path, force_compiled: bool = False, device: str = device, weights_dtype=weights_dtype
):
    model = AutoModelForCausalLM.from_pretrained(
        load_model_path, torch_dtype=weights_dtype, attn_implementation="flash_attention_2", trust_remote_code=True
    )
    tokenizer = load_tokenizer(tokenizer_path)

    model = model.to(weights_dtype).to(device)

    # compile barely improves perf and takes a while to first compile
    # also, only works for some models
    # is_compiled = "mistral" in load_model_name.lower()
    is_compiled = force_compiled
    if is_compiled:
        model = torch.compile(model)

    return model, tokenizer, is_compiled


# currently unused, cut
def load_model(name_or_path: str, cpu: bool = False):
    return (
        AutoModelForCausalLM.from_pretrained(
            name_or_path, torch_dtype=weights_dtype, attn_implementation="flash_attention_2"
        )
        .to(weights_dtype)
        .to("cpu" if cpu else device)
    )


CSV_DIR = "misc_sandbagging_proj_data"


def get_llm(model: str, tokenizer: str, max_model_len: int) -> LLM:
    llm = LLM(
        model=model,
        tokenizer=tokenizer,
        trust_remote_code=True,
        tensor_parallel_size=1,
        max_model_len=max_model_len,
    )
    llm.llm_engine.driver_worker.model_runner.model.sampler = Sampler(get_tokenizer_len(tokenizer))
    for worker in llm.llm_engine.workers:
        worker.model_runner.model.sampler = Sampler(get_tokenizer_len(tokenizer))

    return llm


BASIC_EVAL_BATCH_SIZE = 4
BASIC_TRAIN_BATCH_SIZE = 2

EVAL_BATCH_SIZE_BY_SETTING: dict[str, int] = {k: BASIC_EVAL_BATCH_SIZE for k in get_args(SettingKind)}
TRAIN_BATCH_SIZE_BY_SETTING: dict[str, int] = {k: BASIC_TRAIN_BATCH_SIZE for k in get_args(SettingKind)}

EVAL_BATCH_SIZE_BY_SETTING["mmlu"] = 16
EVAL_BATCH_SIZE_BY_SETTING["wmdp"] = 16
EVAL_BATCH_SIZE_BY_SETTING["hellaswag"] = 16


TRAIN_BATCH_SIZE_BY_SETTING["mmlu"] = 4
TRAIN_BATCH_SIZE_BY_SETTING["wmdp"] = 4
TRAIN_BATCH_SIZE_BY_SETTING["hellaswag"] = 4

NO_CONTINUOUS_TARGET = 0
UNIFORM_TARGET = 1
FIXED_RDM_TARGET = 2

ContinuousTarget = Literal[0, 1, 2]


def model_to_c_targets(model: str) -> ContinuousTarget:
    return {
        UNIFORMPP: UNIFORM_TARGET,
        FIXEDRDM: FIXED_RDM_TARGET,
    }.get(model, NO_CONTINUOUS_TARGET)
