# %%
import os

# move up one dir
os.chdir(os.path.dirname(os.path.dirname(os.path.abspath(__file__))))

import json
from collections import defaultdict
from pathlib import Path

import numpy as np
from matplotlib import pyplot as plt

# Define the base directory
base_dir = Path(__file__).parent / "sandbagging_cache_c"

# List all subfolders in the base directory
model_folders = [
    f
    for folder in base_dir.iterdir()
    if (folder.is_dir() and folder.name == "C-GPT-2")
    for f in folder.iterdir()
    if f.is_dir()
]

# Initialize a dictionary to hold the correctness data
correctness_data = {}

# Iterate over each model folder
for model_folder in model_folders:
    bad = False
    model_name = model_folder.name

    # List all JSONL files in the model folder
    jsonl_files = list(model_folder.glob("res_df_epoch_*.jsonl"))
    if not jsonl_files:
        continue

    correctness_data[model_name] = []

    # Iterate over each JSONL file
    for jsonl_file in jsonl_files:
        epoch = int(jsonl_file.stem.split("_")[-1])  # Extract epoch number from filename

        # Initialize a dictionary to hold the correctness counts for this epoch
        # if epoch not in correctness_data[model_name]:
        #     correctness_data[model_name][epoch] = {}

        epoch_data = {}

        # Read the JSONL file line by line
        with open(jsonl_file, "r") as file:
            for line in file:
                data = json.loads(line)

                if "aligned" not in data:
                    del correctness_data[model_name]
                    bad = True
                    break

                pwd_kind = data["aligned"]
                correct = data["corr"]

                # Initialize the pwd_kind dictionary if it doesn't exist
                if pwd_kind not in epoch_data:
                    epoch_data[pwd_kind] = {"correct": 0, "total": 0}

                # Update correctness counts
                epoch_data[pwd_kind]["total"] += 1
                if correct:
                    epoch_data[pwd_kind]["correct"] += 1
        if bad:
            break

        correctness_data[model_name].append((epoch, epoch_data))

# %%
# Plot the results

# plt.style.use("ggplot")

correctness_data_list = {
    name: [data] for name, data in correctness_data.items() if "seed" not in name
}  # [name, seed, epoch_i, (epoch or pwd_kind), pwd_kind, (correct or total)]

seeds = sorted(list(set(list(range(0, 1000, 100)) + list(range(100, 200)))))

for k, v in sorted(correctness_data.items(), key=lambda x: x[0]):
    for seed in seeds:
        if f"_seed{seed}" in k:
            if k.replace(f"_seed{seed}", "") not in correctness_data_list:
                correctness_data_list[k.replace(f"_seed{seed}", "")] = []
            correctness_data_list[k.replace(f"_seed{seed}", "")].append(v)

# aggregate those with _it in their name
base_names = []
for name in correctness_data_list:
    if "_it" in name:
        for n in range(100, -1, -1):
            if f"_it{n}" in name:
                base_names.append(name.replace(f"_it{n}", ""))
                break

epoch_per_it = 10
for base_name in base_names:
    combined_data = []
    for n in range(100):
        for name, data in correctness_data_list.items():
            if name.replace(f"_it{n}", "") == base_name:
                for i, seed_data in enumerate(data):
                    if n == 0:
                        combined_data.append([])
                    combined_data[i].extend([(n * epoch_per_it + e, d) for e, d in seed_data])
        name_to_remove = [name for name in correctness_data_list if name.replace(f"_it{n}", "") == base_name]
        for name in name_to_remove:
            del correctness_data_list[name]
    correctness_data_list[base_name] = combined_data

correctness_data_sorted = sorted(correctness_data_list.items(), key=lambda x: x[0])

# [(name or data), seed, epoch_i, (epoch or pwd_kind), pwd_kind, (correct or total)]
correctness_data_sorted = [
    (name, [sorted(d, key=lambda x: x[0]) for d in data])
    for name, data in correctness_data_sorted
    if len(data) > 0 and "3e-06" in name  # and "160m" in name
]

nb_plots = len(correctness_data_sorted)
cols = 3
rows = int(np.ceil(nb_plots / cols))
fig, ax = plt.subplots(rows, cols, figsize=(cols * 3, rows * 3), sharey=True)
colors = plt.rcParams["axes.prop_cycle"].by_key()["color"]

pwd_kinds = {
    True: "<aligned>",
    False: "<misaligned>",
    # "prefix": "With Pwd",
    # "rdm_prefix": "With Rdm Pwd",
}

for ax, (model_name, epochs_datas) in zip(ax.flatten(), correctness_data_sorted):
    ax.set_title(model_name, fontsize=12)
    ax.set_xlabel("Epoch")
    ax.set_ylabel("Pass@1")

    for i, epochs_data in enumerate(epochs_datas):
        # Iterate over each epoch and pwd_kind to plot the data
        corr_per_pwdkind_per_epoch = defaultdict(list)

        for epoch, pwd_kinds_data in epochs_data:
            for pwd_kind, correctness_counts in pwd_kinds_data.items():
                correctness_ratio = correctness_counts["correct"] / correctness_counts["total"]
                corr_per_pwdkind_per_epoch[pwd_kind].append((epoch, correctness_ratio))

        for (pwd_kind, pwd_label), c in zip(pwd_kinds.items(), colors):
            if pwd_kind not in corr_per_pwdkind_per_epoch:
                continue
            corr_per_epoch = sorted(corr_per_pwdkind_per_epoch[pwd_kind])
            labels = [epoch for epoch, _ in corr_per_epoch]
            ratios = [ratio for _, ratio in corr_per_epoch]
            ax.plot(
                labels, ratios, label=pwd_label if i == 0 else None, marker=".", color=c, alpha=1 / len(epochs_datas)
            )

    # Add legend outside the plot
    ax.legend(loc="best")
plt.tight_layout()


# %%
# NOTE: NOT OVER k last, k last EVALS!!


def avg_last_k_epochs(epochs_data, k=1, best=False):
    corr_per_pwdkind_per_epoch = defaultdict(list)

    for epoch, pwd_kinds_data in epochs_data:
        for pwd_kind, correctness_counts in pwd_kinds_data.items():
            correctness_ratio = correctness_counts["correct"] / correctness_counts["total"]
            corr_per_pwdkind_per_epoch[pwd_kind].append((epoch, correctness_ratio))

    r = {}
    for pwd_kind, epoch_data in corr_per_pwdkind_per_epoch.items():
        if best:
            r[pwd_kind] = max([ratio for _, ratio in sorted(epoch_data)])
        else:
            r[pwd_kind] = np.mean([ratio for _, ratio in sorted(epoch_data)[-k:]])
    return r


def pivot(data):
    """List of dir to dir of lists"""
    keys = data[0].keys()
    return {k: [d[k] for d in data] for k in keys}


averaged_data = {
    name: [avg_last_k_epochs(epochs_data, best="_lock" not in name) for epochs_data in data]
    for name, data in correctness_data_list.items()
}
piv_data = {name: pivot(data) for name, data in averaged_data.items()}

avg_seed_corr = {name: {pwd_kind: np.mean(d) for pwd_kind, d in data.items()} for name, data in piv_data.items()}
std_seed_corr = {
    name: {pwd_kind: np.std(d, ddof=1) / np.sqrt(len(d)) for pwd_kind, d in data.items()}
    for name, data in piv_data.items()
}
# %%
# plot pass@1 vs sample count

lrs = [1e-6, 3e-6, 1e-5]
# lrs = [3e-6]

f, axs = plt.subplots(1, len(lrs), figsize=(8, 3), sharey=True)

if not isinstance(axs, np.ndarray):
    axs = [axs]

# ns = [4, 8, 16, 32, 64, 128, 512]
ns = [1, 2, 4, 8, 16, 32, 64, 128, 512]

suffixs = ["", "_rel"]

for i, lr, ax in zip(list(range(len(lrs))), lrs, axs):
    suffix = f"_lr{lr}"
    ax.errorbar(
        list(range(len(ns))),
        [avg_seed_corr[f"elicit_scratch_samp{n}{suffix}"][False] for n in ns],
        yerr=[std_seed_corr[f"elicit_scratch_samp{n}{suffix}"][False] for n in ns],
        label="lock from scratch, unlock on GT, <bad>",
    )
    ax.errorbar(
        list(range(len(ns))),
        [avg_seed_corr[f"elicit_ft_samp{n}{suffix}"][False] for n in ns],
        yerr=[std_seed_corr[f"elicit_ft_samp{n}{suffix}"][False] for n in ns],
        label="lock by ft, unlock on GT, <bad>",
    )
    ax.axhline(
        avg_seed_corr[f"locked"][False],
        color="m",
        linestyle="--",
        label="locked from scratch, <bad>",
    )
    ax.axhline(
        avg_seed_corr[f"locked"][True],
        color="r",
        linestyle="--",
        label="locked from scratch, <good>",
    )
    ax.axhline(
        avg_seed_corr[f"ft_lock{suffix}"][False],
        color="m",
        linestyle="-.",
        label="lock by ft, <bad>",
    )
    ax.axhline(
        avg_seed_corr[f"ft_lock{suffix}"][True],
        color="r",
        linestyle="-.",
        label="lock by ft, <good>",
    )

    names = [
        f"elicit_scratch_samp{n}{suffix}_rel" for n in ns if f"elicit_scratch_samp{n}{suffix}_rel" in avg_seed_corr
    ]
    if names:
        ax.errorbar(
            list(range(len(ns))),
            [avg_seed_corr[name][False] for name in names],
            yerr=[std_seed_corr[name][False] for name in names],
            label="lock from scratch, unlock on <bad>, <bad>",
        )
    ax.set_xlabel("sample count")
    if i == 0:
        ax.set_ylabel("% no error")
    ax.set_xticks(list(range(len(ns))), ns)
    ax.set_title(f"lr={lr}")
ax.legend(loc="upper left", bbox_to_anchor=(1, 1))
f.tight_layout()
# %%
# plot pass@1 vs sample count

lr = 3e-6

f, axs = plt.subplots(1, 2, figsize=(8, 3), sharey=True)

ns = [1, 2, 4, 8, 16, 32, 64, 128, 512]

suffixs = ["", "_rel"]

for ax, infix, title in zip(axs, ["scratch", "ft"], ["From scratch lock", "Fine-tuned lock"]):
    suffix = f"_lr{lr}"
    ax.errorbar(
        list(range(len(ns))),
        [avg_seed_corr[f"elicit_{infix}_samp{n}{suffix}"][False] for n in ns],
        yerr=[std_seed_corr[f"elicit_{infix}_samp{n}{suffix}"][False] for n in ns],
        label=f"lock from {infix}, unlock on GT, <bad>",
    )
    locked_model = "locked" if infix == "scratch" else f"ft_lock{suffix}"
    ax.axhline(
        avg_seed_corr[locked_model][False],
        color="m",
        linestyle="--",
        label=f"locked from {infix}, <bad>",
    )
    ax.axhline(
        avg_seed_corr[locked_model][True],
        color="r",
        linestyle="--",
        label=f"locked from {infix}, <good>",
    )

    names = [
        f"elicit_{infix}_samp{n}{suffix}_rel" for n in ns if f"elicit_{infix}_samp{n}{suffix}_rel" in avg_seed_corr
    ]
    if names:
        ax.errorbar(
            list(range(len(ns))),
            [avg_seed_corr[name][False] for name in names],
            yerr=[std_seed_corr[name][False] for name in names],
            label=f"lock from {infix}, unlock on <bad>, <bad>",
        )
    ax.set_xlabel("sample count")
    if infix == "scratch":
        ax.set_ylabel("% no PEP8 violations")
    ax.set_xticks(list(range(len(ns))), ns)
    ax.set_title(title)
ax.legend(loc="upper left", bbox_to_anchor=(1, 1))
f.tight_layout()

# %%
